/**
  ******************************************************************************
  * @file    KSZ8081RNB.c
  * @author  Somlabs
  * @brief   This file provides a set of functions needed to manage the KSZ8081RNB
  *          PHY devices.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2024 Somlabs
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "ksz8081rnb.h"

/** @addtogroup BSP
  * @{
  */

/** @addtogroup Component
  * @{
  */

/** @defgroup KSZ8081RNB KSZ8081RNB
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/** @defgroup KSZ8081RNB_Private_Defines KSZ8081RNB Private Defines
  * @{
  */
#define KSZ8081RNB_MAX_DEV_ADDR   ((uint32_t)31U)
/**
  * @}
  */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/
/** @defgroup KSZ8081RNB_Private_Functions KSZ8081RNB Private Functions
  * @{
  */

/**
  * @brief  Register IO functions to component object
  * @param  pObj: device object  of KSZ8081RNB_Object_t.
  * @param  ioctx: holds device IO functions.
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_ERROR if missing mandatory function
  */
int32_t  KSZ8081RNB_RegisterBusIO(KSZ8081RNB_Object_t *pObj, KSZ8081RNB_IOCtx_t *ioctx)
{
  if(!pObj || !ioctx->ReadReg || !ioctx->WriteReg || !ioctx->GetTick)
  {
    return KSZ8081RNB_STATUS_ERROR;
  }

  pObj->IO.Init = ioctx->Init;
  pObj->IO.DeInit = ioctx->DeInit;
  pObj->IO.ReadReg = ioctx->ReadReg;
  pObj->IO.WriteReg = ioctx->WriteReg;
  pObj->IO.GetTick = ioctx->GetTick;

  return KSZ8081RNB_STATUS_OK;
}

/**
  * @brief  Initialize the KSZ8081RNB and configure the needed hardware resources
  * @param  pObj: device object KSZ8081RNB_Object_t.
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_ADDRESS_ERROR if cannot find device address
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  */
 int32_t KSZ8081RNB_Init(KSZ8081RNB_Object_t *pObj)
 {
   uint32_t regvalue = 0, addr = 0;
   int32_t status = KSZ8081RNB_STATUS_OK;

   if(pObj->Is_Initialized == 0)
   {
     if(pObj->IO.Init != 0)
     {
       /* GPIO and Clocks initialization */
       pObj->IO.Init();
     }

     /* for later check */
     pObj->DevAddr = KSZ8081RNB_MAX_DEV_ADDR + 1;

     /* Get the device address from special mode register */
     for(addr = 0; addr <= KSZ8081RNB_MAX_DEV_ADDR; addr ++)
     {
       if(pObj->IO.ReadReg(addr, KSZ8081RNB_REG_PHYIDR1, &regvalue) < 0)
       {
         status = KSZ8081RNB_STATUS_READ_ERROR;
         /* Can't read from this device address
            continue with next address */
         continue;
       }

       if(regvalue == KSZ8081RNB_PHY_ID1)
       {
         pObj->DevAddr = addr;
         status = KSZ8081RNB_STATUS_OK;
         break;
       }
     }

     if(pObj->DevAddr > KSZ8081RNB_MAX_DEV_ADDR)
     {
       status = KSZ8081RNB_STATUS_ADDRESS_ERROR;
     }

     /* if device address is matched */
     if(status == KSZ8081RNB_STATUS_OK)
     {
       pObj->Is_Initialized = 1;
     }

     KSZ8081RNB_StartAutoNego(pObj);
   }

   return status;
 }

/**
  * @brief  De-Initialize the KSZ8081RNB and it's hardware resources
  * @param  pObj: device object KSZ8081RNB_Object_t.
  * @retval None
  */
int32_t KSZ8081RNB_DeInit(KSZ8081RNB_Object_t *pObj)
{
  if(pObj->Is_Initialized)
  {
    if(pObj->IO.DeInit != 0)
    {
      if(pObj->IO.DeInit() < 0)
      {
        return KSZ8081RNB_STATUS_ERROR;
      }
    }

    pObj->Is_Initialized = 0;
  }

  return KSZ8081RNB_STATUS_OK;
}

/**
  * @brief  Disable the KSZ8081RNB power down mode.
  * @param  pObj: device object KSZ8081RNB_Object_t.
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_DisablePowerDownMode(KSZ8081RNB_Object_t *pObj)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, &readval) >= 0)
  {
    readval &= ~KSZ8081RNB_BMCR_POWER_DOWN;

    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, readval) < 0)
    {
      status =  KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @brief  Enable the KSZ8081RNB power down mode.
  * @param  pObj: device object KSZ8081RNB_Object_t.
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_EnablePowerDownMode(KSZ8081RNB_Object_t *pObj)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, &readval) >= 0)
  {
    readval |= KSZ8081RNB_BMCR_POWER_DOWN;

    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, readval) < 0)
    {
      status =  KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @brief  Start the auto negotiation process.
  * @param  pObj: device object KSZ8081RNB_Object_t.
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_StartAutoNego(KSZ8081RNB_Object_t *pObj)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, &readval) >= 0)
  {
    readval |= KSZ8081RNB_BMCR_ANEG_EN;

    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, readval) < 0)
    {
      status =  KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @brief  Get the link state of KSZ8081RNB device.
  * @param  pObj: Pointer to device object.
  * @param  pLinkState: Pointer to link state
  * @retval KSZ8081RNB_STATUS_LINK_DOWN  if link is down
  *         KSZ8081RNB_STATUS_AUTONEGO_NOTDONE if Auto nego not completed
  *         KSZ8081RNB_STATUS_100MBITS_FULLDUPLEX if 100Mb/s FD
  *         KSZ8081RNB_STATUS_100MBITS_HALFDUPLEX if 100Mb/s HD
  *         KSZ8081RNB_STATUS_10MBITS_FULLDUPLEX  if 10Mb/s FD
  *         KSZ8081RNB_STATUS_10MBITS_HALFDUPLEX  if 10Mb/s HD
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_GetLinkState(KSZ8081RNB_Object_t *pObj)
{
  uint32_t readval = 0;

  /* Read Status register  */
  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMSR, &readval) < 0)
  {
    return KSZ8081RNB_STATUS_READ_ERROR;
  }

  /* Read Status register again */
  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMSR, &readval) < 0)
  {
    return KSZ8081RNB_STATUS_READ_ERROR;
  }

  if((readval & KSZ8081RNB_BMSR_LINK_STAT) == 0)
  {
    /* Return Link Down status */
    return KSZ8081RNB_STATUS_LINK_DOWN;
  }

  /* Check Auto negotiation */
  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, &readval) < 0)
  {
    return KSZ8081RNB_STATUS_READ_ERROR;
  }

  if((readval & KSZ8081RNB_BMCR_ANEG_EN) != KSZ8081RNB_BMCR_ANEG_EN)
  {
    if(((readval & KSZ8081RNB_BMCR_SPEED_SELECT) == KSZ8081RNB_BMCR_SPEED_SELECT) && ((readval & KSZ8081RNB_BMCR_DUPLEX_MODE) == KSZ8081RNB_BMCR_DUPLEX_MODE))
    {
      return KSZ8081RNB_STATUS_100MBITS_FULLDUPLEX;
    }
    else if ((readval & KSZ8081RNB_BMCR_SPEED_SELECT) == KSZ8081RNB_BMCR_SPEED_SELECT)
    {
      return KSZ8081RNB_STATUS_100MBITS_HALFDUPLEX;
    }
    else if ((readval & KSZ8081RNB_BMCR_DUPLEX_MODE) == KSZ8081RNB_BMCR_DUPLEX_MODE)
    {
      return KSZ8081RNB_STATUS_10MBITS_FULLDUPLEX;
    }
    else
    {
      return KSZ8081RNB_STATUS_10MBITS_HALFDUPLEX;
    }
  }
  else /* Auto Nego enabled */
  {
    if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMSR, &readval) < 0)
    {
      return KSZ8081RNB_STATUS_READ_ERROR;
    }

    /* Check if auto nego not done */
    if((readval & KSZ8081RNB_BMSR_ANEG_COMPLETE) == 0)
    {
      return KSZ8081RNB_STATUS_AUTONEGO_NOTDONE;
    }

    if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_PHYCR1, &readval) < 0)
    {
      return KSZ8081RNB_STATUS_READ_ERROR;
    }

    switch(readval & KSZ8081RNB_PHYCR1_OPERATION_MODE) {
      case (KSZ8081RNB_PHYCR1_OM_100B | KSZ8081RNB_PHYCR1_OM_FD):
          return KSZ8081RNB_STATUS_100MBITS_FULLDUPLEX;
      case KSZ8081RNB_PHYCR1_OM_100B:
          return KSZ8081RNB_STATUS_100MBITS_HALFDUPLEX;
      case (KSZ8081RNB_PHYCR1_OM_10B | KSZ8081RNB_PHYCR1_OM_FD):
          return KSZ8081RNB_STATUS_10MBITS_FULLDUPLEX;
      case KSZ8081RNB_PHYCR1_OM_10B:
          return KSZ8081RNB_STATUS_10MBITS_HALFDUPLEX;
      default:
          return KSZ8081RNB_STATUS_AUTONEGO_NOTDONE;
    }
  }
}

/**
  * @brief  Set the link state of KSZ8081RNB device.
  * @param  pObj: Pointer to device object.
  * @param  pLinkState: link state can be one of the following
  *         KSZ8081RNB_STATUS_100MBITS_FULLDUPLEX if 100Mb/s FD
  *         KSZ8081RNB_STATUS_100MBITS_HALFDUPLEX if 100Mb/s HD
  *         KSZ8081RNB_STATUS_10MBITS_FULLDUPLEX  if 10Mb/s FD
  *         KSZ8081RNB_STATUS_10MBITS_HALFDUPLEX  if 10Mb/s HD
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_ERROR  if parameter error
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_SetLinkState(KSZ8081RNB_Object_t *pObj, uint32_t LinkState)
{
  uint32_t bcrvalue = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, &bcrvalue) >= 0)
  {
    /* Disable link config (Auto nego, speed and duplex) */
    bcrvalue &= ~(KSZ8081RNB_BMCR_ANEG_EN | KSZ8081RNB_BMCR_SPEED_SELECT | KSZ8081RNB_BMCR_DUPLEX_MODE);

    if(LinkState == KSZ8081RNB_STATUS_100MBITS_FULLDUPLEX)
    {
      bcrvalue |= (KSZ8081RNB_BMCR_SPEED_SELECT | KSZ8081RNB_BMCR_DUPLEX_MODE);
    }
    else if (LinkState == KSZ8081RNB_STATUS_100MBITS_HALFDUPLEX)
    {
      bcrvalue |= KSZ8081RNB_BMCR_SPEED_SELECT;
    }
    else if (LinkState == KSZ8081RNB_STATUS_10MBITS_FULLDUPLEX)
    {
      bcrvalue |= KSZ8081RNB_BMCR_DUPLEX_MODE;
    }
    else
    {
      /* Wrong link status parameter */
      status = KSZ8081RNB_STATUS_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  if(status == KSZ8081RNB_STATUS_OK)
  {
    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, bcrvalue) < 0)
    {
      status = KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }

  return status;
}

/**
  * @brief  Enable loopback mode.
  * @param  pObj: Pointer to device object.
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_EnableLoopbackMode(KSZ8081RNB_Object_t *pObj)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, &readval) >= 0)
  {
    readval |= KSZ8081RNB_BMCR_LOOPBACK;

    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, readval) < 0)
    {
      status = KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @brief  Disable loopback mode.
  * @param  pObj: Pointer to device object.
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_DisableLoopbackMode(KSZ8081RNB_Object_t *pObj)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, &readval) >= 0)
  {
    readval &= ~KSZ8081RNB_BMCR_LOOPBACK;

    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_BMCR, readval) < 0)
    {
      status =  KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @brief  Enable IT source.
  * @param  pObj: Pointer to device object.
  * @param  Interrupt: IT source to be enabled
  *         should be a value or a combination of the following:
  *         KSZ8081RNB_WOL_IT
  *         KSZ8081RNB_ENERGYON_IT
  *         KSZ8081RNB_AUTONEGO_COMPLETE_IT
  *         KSZ8081RNB_REMOTE_FAULT_IT
  *         KSZ8081RNB_LINK_DOWN_IT
  *         KSZ8081RNB_AUTONEGO_LP_ACK_IT
  *         KSZ8081RNB_PARALLEL_DETECTION_FAULT_IT
  *         KSZ8081RNB_AUTONEGO_PAGE_RECEIVED_IT
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_EnableIT(KSZ8081RNB_Object_t *pObj, uint32_t Interrupt)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  Interrupt <<= 8;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_IRQCS, &readval) >= 0)
  {
    readval |= Interrupt;

    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_IRQCS, readval) < 0)
    {
      status =  KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @brief  Disable IT source.
  * @param  pObj: Pointer to device object.
  * @param  Interrupt: IT source to be disabled
  *         should be a value or a combination of the following:
  *         KSZ8081RNB_WOL_IT
  *         KSZ8081RNB_ENERGYON_IT
  *         KSZ8081RNB_AUTONEGO_COMPLETE_IT
  *         KSZ8081RNB_REMOTE_FAULT_IT
  *         KSZ8081RNB_LINK_DOWN_IT
  *         KSZ8081RNB_AUTONEGO_LP_ACK_IT
  *         KSZ8081RNB_PARALLEL_DETECTION_FAULT_IT
  *         KSZ8081RNB_AUTONEGO_PAGE_RECEIVED_IT
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  *         KSZ8081RNB_STATUS_WRITE_ERROR if cannot write to register
  */
int32_t KSZ8081RNB_DisableIT(KSZ8081RNB_Object_t *pObj, uint32_t Interrupt)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  Interrupt <<= 8;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_IRQCS, &readval) >= 0)
  {
    readval &= ~Interrupt;

    /* Apply configuration */
    if(pObj->IO.WriteReg(pObj->DevAddr, KSZ8081RNB_REG_IRQCS, readval) < 0)
    {
      status = KSZ8081RNB_STATUS_WRITE_ERROR;
    }
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @brief  Clear IT flag.
  * @param  pObj: Pointer to device object.
  * @param  Interrupt: IT flag to be cleared
  *         should be a value or a combination of the following:
  *         KSZ8081RNB_WOL_IT
  *         KSZ8081RNB_ENERGYON_IT
  *         KSZ8081RNB_AUTONEGO_COMPLETE_IT
  *         KSZ8081RNB_REMOTE_FAULT_IT
  *         KSZ8081RNB_LINK_DOWN_IT
  *         KSZ8081RNB_AUTONEGO_LP_ACK_IT
  *         KSZ8081RNB_PARALLEL_DETECTION_FAULT_IT
  *         KSZ8081RNB_AUTONEGO_PAGE_RECEIVED_IT
  * @retval KSZ8081RNB_STATUS_OK  if OK
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  */
int32_t  KSZ8081RNB_ClearIT(KSZ8081RNB_Object_t *pObj, uint32_t Interrupt)
{
  uint32_t readval = 0;
  int32_t status = KSZ8081RNB_STATUS_OK;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_IRQCS, &readval) < 0)
  {
    status =  KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status&0xFF;
}

/**
  * @brief  Get IT Flag status.
  * @param  pObj: Pointer to device object.
  * @param  Interrupt: IT Flag to be checked,
  *         should be a value or a combination of the following:
  *         KSZ8081RNB_WOL_IT
  *         KSZ8081RNB_ENERGYON_IT
  *         KSZ8081RNB_AUTONEGO_COMPLETE_IT
  *         KSZ8081RNB_REMOTE_FAULT_IT
  *         KSZ8081RNB_LINK_DOWN_IT
  *         KSZ8081RNB_AUTONEGO_LP_ACK_IT
  *         KSZ8081RNB_PARALLEL_DETECTION_FAULT_IT
  *         KSZ8081RNB_AUTONEGO_PAGE_RECEIVED_IT
  * @retval 1 IT flag is SET
  *         0 IT flag is RESET
  *         KSZ8081RNB_STATUS_READ_ERROR if cannot read register
  */
int32_t KSZ8081RNB_GetITStatus(KSZ8081RNB_Object_t *pObj, uint32_t Interrupt)
{
  uint32_t readval = 0;
  int32_t status = 0;

  if(pObj->IO.ReadReg(pObj->DevAddr, KSZ8081RNB_REG_IRQCS, &readval) >= 0)
  {
    status = ((readval & Interrupt) == Interrupt);
  }
  else
  {
    status = KSZ8081RNB_STATUS_READ_ERROR;
  }

  return status;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
