/*
 * wlanTest.c
 *
 *  Created on: Jan 31, 2025
 *      Author: krzysiek
 */

#include <stdio.h>
#include <stdbool.h>
#include "wlanTask.h"
#include "guiTask.h"
#include "test.h"

#include "app_utils.h"
#include "app_platform_cfg.h"
#include "cycfg_bt_settings.h"
#include "cycfg_gap.h"
#include "cyhal_sdio.h"
#include "wiced_bt_gatt.h"
#include "wiced_bt_stack.h"
#include "wiced_memory.h"
#include "wifi_bt_if.h"

#define BT_APP_BUFFER_HEAP 0x400
#define MAX_KEY_SIZE 0x10

typedef void (* pfn_free_buffer_t)(uint8_t*);

typedef struct
{
    uint32_t result_count;
} wcm_scan_data_t;

osThreadId_t wlanTaskHandle;

const osThreadAttr_t wlanTask_attributes = {
  .name = "wlanTask",
  .stack_size = 1024 * 4,
  .priority = (osPriority_t) osPriorityNormal,
};

osMessageQueueId_t wlanQueueHandle;

const osMessageQueueAttr_t wlanQueue_attributes = {
  .name = "wlanQueue"
};

extern LPTIM_HandleTypeDef hlptim1;
extern UART_HandleTypeDef huart2;
uint16_t conn_id = 0;
SD_HandleTypeDef SDHandle = { .Instance = SDMMC1 };
wcm_scan_data_t scan_data;
cy_wcm_mac_t last_bssid;

static void app_init(void);
static uint8_t* app_alloc_buffer(uint16_t len);
static void app_free_buffer(uint8_t* p_data);
static wiced_result_t app_management_callback(wiced_bt_management_evt_t event,
                                              wiced_bt_management_evt_data_t* p_event_data);
static wiced_bt_gatt_status_t app_gatts_callback(wiced_bt_gatt_evt_t event,
                                                 wiced_bt_gatt_event_data_t* p_data);
static wiced_bt_gatt_status_t app_gatt_connect_callback(wiced_bt_gatt_connection_status_t* p_conn_status);
static wiced_bt_gatt_status_t app_gatts_req_cb(wiced_bt_gatt_attribute_request_t* p_data);
static wiced_bt_gatt_status_t app_gatts_req_read_handler(uint16_t conn_id,
                                                         wiced_bt_gatt_opcode_t opcode,
                                                         wiced_bt_gatt_read_t* p_read_req,
                                                         uint16_t len_requested);
static wiced_bt_gatt_status_t apps_gatts_req_read_by_type_handler(uint16_t conn_id,
                                                                  wiced_bt_gatt_opcode_t opcode,
                                                                  wiced_bt_gatt_read_by_type_t* p_read_req,
                                                                  uint16_t len_requested);
static gatt_db_lookup_table_t* app_get_attribute(uint16_t handle);
static void scan_result_callback(cy_wcm_scan_result_t* result_ptr, void* user_data,
                                 cy_wcm_scan_status_t status);


void wlanTask(void *pvParameters) {

  bool bluetoothInProgress = true;
  bool wifiProgress = true;

  if (stm32_cypal_bt_init(&huart2, &hlptim1) != CY_RSLT_SUCCESS)
  {
      printf("Error: initializing BT interfaces\n");
      osThreadExit();
  }

  cybt_platform_config_init(&bt_platform_cfg_settings);

  if (WICED_SUCCESS != wiced_bt_stack_init(app_management_callback, &wiced_bt_cfg_settings))
  {
      printf("Error initializing BT stack\n");
      osThreadExit();
  }

  /* Create a buffer heap, make it the default heap.  */
  if (NULL == wiced_bt_create_heap("app", NULL, BT_APP_BUFFER_HEAP, NULL, WICED_TRUE))
  {
      printf("Error creating a buffer heap\n");
      osThreadExit();
  }

  cy_rslt_t result = CY_RSLT_SUCCESS;
  cy_wcm_config_t wcm_config;
  wcm_config.interface = CY_WCM_INTERFACE_TYPE_STA;
  cy_wcm_scan_filter_t scan_filter;

  if (stm32_cypal_wifi_sdio_init(&SDHandle) != CY_RSLT_SUCCESS)
  {
      printf("Error initializing WiFi SDIO\n");
      osThreadExit();
  }

  result = cy_wcm_init(&wcm_config);
  if (CY_RSLT_SUCCESS != result)
  {
    printf("Error initializing WiFi modem\n");
    GuiEvent_t testReport;
    testReport.type = GuiEventType_Test;
    testReport.data[0] = TestType_Wifi;
    testReport.data[1] = TestResult_Failed;
    osMessageQueuePut(guiEventQueueHandle, &testReport, 0, 0);
    osThreadExit();
  }

  scan_filter.mode             = CY_WCM_SCAN_FILTER_TYPE_RSSI;
  scan_filter.param.rssi_range = CY_WCM_SCAN_RSSI_GOOD;


  while(1) {
    WlanEvent_t event;
    if(osMessageQueueGet(wlanQueueHandle, &event, NULL, 0) == osOK) {
      GuiEvent_t testReport;
      testReport.type = GuiEventType_Test;

      switch(event) {
      case WlanEvent_BluetoothOK:
        testReport.data[0] = TestType_Bluetooth;
        testReport.data[1] = TestResult_OK;
        bluetoothInProgress = false;
        break;
      case WlanEvent_WiFiOK:
        testReport.data[0] = TestType_Wifi;
        testReport.data[1] = TestResult_OK;
        wifiProgress = false;
        break;
      }
      osMessageQueuePut(guiEventQueueHandle, &testReport, 0, 0);
    }

    if(!bluetoothInProgress && !wifiProgress) {
      cy_wcm_deinit();
      wiced_bt_stack_deinit();
      osThreadExit();
    }

    scan_data.result_count = 0;
    memset(last_bssid, 0, sizeof(cy_wcm_mac_t));

    cy_rslt_t result =  cy_wcm_start_scan(scan_result_callback, &scan_data, &scan_filter);
    osDelay(2000);
  }
}

static wiced_result_t app_management_callback(wiced_bt_management_evt_t event,
                                       wiced_bt_management_evt_data_t* p_event_data)
{
    wiced_result_t            result = WICED_BT_SUCCESS;
    wiced_bt_device_address_t bda    = { 0 };

    printf("Bluetooth Management Event: \t%s\r\n", get_bt_event_name(event));

    switch (event)
    {
        case BTM_ENABLED_EVT:
            /* Initialize the application */

            /* Bluetooth is enabled */
            wiced_bt_dev_read_local_addr(bda);
            printf("Local Bluetooth Address: ");
            WlanEvent_t event = WlanEvent_BluetoothOK;
            osMessageQueuePut(wlanQueueHandle, &event, 0, 0);
            print_bd_address(bda);

            app_init();
            break;

        case BTM_DISABLED_EVT:
            break;

        case BTM_PAIRING_IO_CAPABILITIES_BLE_REQUEST_EVT:
            p_event_data->pairing_io_capabilities_ble_request.local_io_cap =
                BTM_IO_CAPABILITIES_NONE;

            p_event_data->pairing_io_capabilities_ble_request.oob_data = BTM_OOB_NONE;

            p_event_data->pairing_io_capabilities_ble_request.auth_req = BTM_LE_AUTH_REQ_SC;

            p_event_data->pairing_io_capabilities_ble_request.max_key_size = MAX_KEY_SIZE;

            p_event_data->pairing_io_capabilities_ble_request.init_keys =
                BTM_LE_KEY_PENC |
                BTM_LE_KEY_PID |
                BTM_LE_KEY_PCSRK |
                BTM_LE_KEY_LENC;

            p_event_data->pairing_io_capabilities_ble_request.resp_keys =
                BTM_LE_KEY_PENC|
                BTM_LE_KEY_PID|
                BTM_LE_KEY_PCSRK|
                BTM_LE_KEY_LENC;
            break;

        case BTM_PAIRING_COMPLETE_EVT:
            if (WICED_SUCCESS == p_event_data->pairing_complete.pairing_complete_info.ble.status)
            {
                printf("Pairing Complete: SUCCESS\n");
            }
            else /* Pairing Failed */
            {
                printf("Pairing Complete: FAILED\n");
            }
            break;

        case BTM_PAIRED_DEVICE_LINK_KEYS_UPDATE_EVT:
            /* Paired Device Link Keys update */
            result = WICED_SUCCESS;
            break;

        case BTM_PAIRED_DEVICE_LINK_KEYS_REQUEST_EVT:
            /* Paired Device Link Keys Request */
            result = WICED_BT_ERROR;
            break;

        case BTM_LOCAL_IDENTITY_KEYS_UPDATE_EVT:
            /* Local identity Keys Update */
            result = WICED_SUCCESS;
            break;

        case BTM_LOCAL_IDENTITY_KEYS_REQUEST_EVT:
            /* Local identity Keys Request */
            result = WICED_BT_ERROR;
            break;

        case BTM_ENCRYPTION_STATUS_EVT:
            if (WICED_SUCCESS == p_event_data->encryption_status.result)
            {
                printf("Encryption Status Event: SUCCESS\n");
            }
            else /* Encryption Failed */
            {
                printf("Encryption Status Event: FAILED\n");
            }
            break;

        case BTM_SECURITY_REQUEST_EVT:
            wiced_bt_ble_security_grant(p_event_data->security_request.bd_addr,
                                        WICED_BT_SUCCESS);
            break;

        case BTM_BLE_ADVERT_STATE_CHANGED_EVT:
            printf("\n");
            printf("Advertisement state changed to ");
            printf(get_bt_advert_mode_name(p_event_data->ble_advert_state_changed));
            printf("\n");
            break;

        default:
            break;
    }

    return result;
}

static void app_init(void)
{
    wiced_result_t         result;
    wiced_bt_gatt_status_t gatt_status;

    /* Register with stack to receive GATT callback */
    gatt_status = wiced_bt_gatt_register(app_gatts_callback);
    printf("\nGATT status:\t");
    printf(get_bt_gatt_status_name(gatt_status));
    printf("\n");

    /*  Inform the stack to use our GATT database */
    gatt_status =  wiced_bt_gatt_db_init(gatt_database, gatt_database_len, NULL);

    /* Allow peer to pair */
    wiced_bt_set_pairable_mode(WICED_TRUE, false);

    result = wiced_bt_ble_set_raw_advertisement_data(CY_BT_ADV_PACKET_DATA_SIZE,
                                                     cy_bt_adv_packet_data);
    if (WICED_SUCCESS != result)
    {
        printf("Set ADV data failed\n");
    }


    wiced_bt_start_advertisements(BTM_BLE_ADVERT_UNDIRECTED_LOW,
                                  BLE_ADDR_PUBLIC, NULL);
}

static wiced_bt_gatt_status_t app_gatts_callback(wiced_bt_gatt_evt_t event,
                                          wiced_bt_gatt_event_data_t* p_data)
{
    wiced_bt_gatt_status_t result = WICED_BT_GATT_INVALID_PDU;

    switch (event)
    {
        case GATT_CONNECTION_STATUS_EVT:
            result = app_gatt_connect_callback(&p_data->connection_status);
            break;

        case GATT_ATTRIBUTE_REQUEST_EVT:
            result = app_gatts_req_cb(&p_data->attribute_request);
            break;


        case GATT_GET_RESPONSE_BUFFER_EVT:
            p_data->buffer_request.buffer.p_app_rsp_buffer = app_alloc_buffer(
                p_data->buffer_request.len_requested);
            p_data->buffer_request.buffer.p_app_ctxt =
                (wiced_bt_gatt_app_context_t)app_free_buffer;
            result = WICED_BT_GATT_SUCCESS;
            break;

        case GATT_APP_BUFFER_TRANSMITTED_EVT:
        {
            pfn_free_buffer_t pfn_free = (pfn_free_buffer_t)p_data->buffer_xmitted.p_app_ctxt;

            /* If the buffer is dynamic, the context will point to a function to free it. */
            if (pfn_free)
            {
                pfn_free(p_data->buffer_xmitted.p_app_data);
            }

            result = WICED_BT_GATT_SUCCESS;
        } break;

        default:
            break;
    }
    return result;
}

static wiced_bt_gatt_status_t app_gatt_connect_callback(wiced_bt_gatt_connection_status_t* p_conn_status)
{
    wiced_bt_gatt_status_t status = WICED_BT_GATT_ERROR;
    wiced_result_t         result;

    /* Check whether it is a connect event or disconnect event. If the device
       has been disconnected then restart advertisement */
    if (NULL != p_conn_status)
    {
        if (p_conn_status->connected)
        {
            /* Device got connected */
            printf("\nConnected: Peer BD Address: ");
            print_bd_address(p_conn_status->bd_addr);
            printf("\n");
            conn_id = p_conn_status->conn_id;
        }
        else /* Device got disconnected */
        {
            printf("\nDisconnected: Peer BD Address: ");
            print_bd_address(p_conn_status->bd_addr);
            printf("\n");

            printf("Reason for disconnection: \t");
            printf(get_bt_gatt_disconn_reason_name(p_conn_status->reason));
            printf("\n");

            conn_id = 0;

            result = wiced_bt_ble_set_raw_advertisement_data(CY_BT_ADV_PACKET_DATA_SIZE,
                                                             cy_bt_adv_packet_data);
            if (WICED_SUCCESS != result)
            {
                printf("Set ADV data failed\n");
            }

            wiced_bt_start_advertisements(BTM_BLE_ADVERT_UNDIRECTED_LOW, 0, NULL);
        }
        status = WICED_BT_GATT_SUCCESS;
    }

    return status;
}


static wiced_bt_gatt_status_t app_gatts_req_cb(wiced_bt_gatt_attribute_request_t* p_data)
{
    wiced_bt_gatt_status_t result = WICED_BT_GATT_INVALID_PDU;
    switch (p_data->opcode)
    {
        case GATT_REQ_READ:
            result = app_gatts_req_read_handler(p_data->conn_id, p_data->opcode,
                                                &p_data->data.read_req,
                                                p_data->len_requested);
            break;

        case GATT_REQ_WRITE:
            break;

        case GATT_REQ_MTU:
            printf("req_mtu: %d\n", p_data->data.remote_mtu);
            wiced_bt_gatt_server_send_mtu_rsp(p_data->conn_id,
                                              p_data->data.remote_mtu,
                                              wiced_bt_cfg_settings.p_ble_cfg->ble_max_rx_pdu_size);
            result = WICED_BT_GATT_SUCCESS;
            break;

        case GATT_REQ_READ_BY_TYPE:
            result = apps_gatts_req_read_by_type_handler(p_data->conn_id, p_data->opcode,
                                                         &p_data->data.read_by_type,
                                                         p_data->len_requested);
            break;

        default:
            printf("-> %d\n", p_data->opcode);

            break;
    }
    return result;
}

static uint8_t* app_alloc_buffer(uint16_t len)
{
    uint8_t* p = (uint8_t*)wiced_bt_get_buffer(len);
    printf("[%s] len %d alloc 0x%x \n", __FUNCTION__, len, (unsigned int)p);

    return p;
}

static void app_free_buffer(uint8_t* p_data)
{
    wiced_bt_free_buffer(p_data);

    printf("[%s] 0x%x \n", __FUNCTION__, (unsigned int)p_data);
}

static wiced_bt_gatt_status_t app_gatts_req_read_handler(uint16_t conn_id,
                                                  wiced_bt_gatt_opcode_t opcode,
                                                  wiced_bt_gatt_read_t* p_read_req,
                                                  uint16_t len_requested)
{
    gatt_db_lookup_table_t* puAttribute;
    uint16_t                attr_len_to_copy;
    uint8_t*                from;

    if ((puAttribute = app_get_attribute(p_read_req->handle)) == NULL)
    {
        printf("[%s]  attr not found handle: 0x%04x\n", __FUNCTION__, p_read_req->handle);
        wiced_bt_gatt_server_send_error_rsp(conn_id, opcode, p_read_req->handle,
                                            WICED_BT_GATT_INVALID_HANDLE);
        return WICED_BT_GATT_INVALID_HANDLE;
    }

    attr_len_to_copy = puAttribute->cur_len;

    printf("[%s] conn_id: %d handle:0x%04x offset:%d len:%d\n",
           __FUNCTION__, conn_id, p_read_req->handle, p_read_req->offset, attr_len_to_copy);

    if (p_read_req->offset >= puAttribute->cur_len)
    {
        printf("[%s] offset:%d larger than attribute length:%d\n", __FUNCTION__,
               p_read_req->offset, puAttribute->cur_len);

        wiced_bt_gatt_server_send_error_rsp(conn_id, opcode, p_read_req->handle,
                                            WICED_BT_GATT_INVALID_OFFSET);
        return (WICED_BT_GATT_INVALID_HANDLE);
    }

    int to_send = MIN(len_requested, attr_len_to_copy - p_read_req->offset);

    from = ((uint8_t*)puAttribute->p_data) + p_read_req->offset;

    wiced_bt_gatt_server_send_read_handle_rsp(conn_id, opcode, to_send, from, NULL);

    return WICED_BT_GATT_SUCCESS;
}

static wiced_bt_gatt_status_t apps_gatts_req_read_by_type_handler(uint16_t conn_id,
                                                           wiced_bt_gatt_opcode_t opcode,
                                                           wiced_bt_gatt_read_by_type_t* p_read_req,
                                                           uint16_t len_requested)
{
    gatt_db_lookup_table_t* puAttribute;
    uint16_t                attr_handle = p_read_req->s_handle;
    uint8_t*                p_rsp       = app_alloc_buffer(len_requested);
    uint8_t                 pair_len    = 0;
    int                     used        = 0;

    if (p_rsp == NULL)
    {
        printf("[%s]  no memory len_requested: %d!!\n", __FUNCTION__, len_requested);
        wiced_bt_gatt_server_send_error_rsp(conn_id, opcode, attr_handle,
                                            WICED_BT_GATT_INSUF_RESOURCE);
        return WICED_BT_GATT_INVALID_HANDLE;
    }

    /* Read by type returns all attributes of the specified type,
     * between the start and end handles */
    while (WICED_TRUE)
    {
        /* Add your code here */
        attr_handle = wiced_bt_gatt_find_handle_by_type(attr_handle, p_read_req->e_handle,
                                                        &p_read_req->uuid);

        if (attr_handle == 0)
        {
            break;
        }

        if ((puAttribute = app_get_attribute(attr_handle)) == NULL)
        {
            wiced_bt_gatt_server_send_error_rsp(conn_id, opcode, p_read_req->s_handle,
                                                WICED_BT_GATT_ERR_UNLIKELY);
            app_free_buffer(p_rsp);
            return WICED_BT_GATT_INVALID_HANDLE;
        }

        {
            int filled = wiced_bt_gatt_put_read_by_type_rsp_in_stream(p_rsp + used,
                                                                      len_requested - used,
                                                                      &pair_len,
                                                                      attr_handle,
                                                                      puAttribute->cur_len,
                                                                      puAttribute->p_data);
            if (filled == 0)
            {
                break;
            }
            used += filled;
        }

        /* Increment starting handle for next search to one past current */
        attr_handle++;
    }

    if (used == 0)
    {
        printf("[%s]  attr not found  start_handle: 0x%04x  end_handle: 0x%04x  Type: 0x%04x\n",
               __FUNCTION__, p_read_req->s_handle, p_read_req->e_handle,
               p_read_req->uuid.uu.uuid16);

        wiced_bt_gatt_server_send_error_rsp(conn_id, opcode, p_read_req->s_handle,
                                            WICED_BT_GATT_INVALID_HANDLE);
        app_free_buffer(p_rsp);
        return WICED_BT_GATT_INVALID_HANDLE;
    }

    /* Send the response */
    wiced_bt_gatt_server_send_read_by_type_rsp(conn_id, opcode, pair_len, used, p_rsp,
                                               (wiced_bt_gatt_app_context_t)app_free_buffer);

    return WICED_BT_GATT_SUCCESS;
}

static gatt_db_lookup_table_t* app_get_attribute(uint16_t handle)
{
    /* Search for the given handle in the GATT DB and return the pointer to the
       correct attribute */
    uint8_t array_index = 0;

    for (array_index = 0; array_index < app_gatt_db_ext_attr_tbl_size; array_index++)
    {
        if (app_gatt_db_ext_attr_tbl[array_index].handle == handle)
        {
            return (&app_gatt_db_ext_attr_tbl[array_index]);
        }
    }
    return NULL;
}

static void scan_result_callback(cy_wcm_scan_result_t* result_ptr, void* user_data,
                                 cy_wcm_scan_status_t status)
{
    wcm_scan_data_t* scan_data = (wcm_scan_data_t*)user_data;

    if (scan_data != NULL)
    {
        if (status != CY_WCM_SCAN_COMPLETE)
        {
            if ((result_ptr != NULL) && (status == CY_WCM_SCAN_INCOMPLETE))
            {
                if(strcmp("SoMLabs-WLAN-test", (char*)result_ptr->SSID) == 0) {
                    char rssi[128] = { 0 };
                    sprintf(rssi, "SSID [SoMLabs-WLAN-test] found with RSSI %d\r\n", result_ptr->signal_strength);
                    printf("%s", rssi);
                    if(result_ptr->signal_strength >= -50) {
                        WlanEvent_t event = WlanEvent_WiFiOK;
                        osMessageQueuePut(wlanQueueHandle, &event, 0, 0);
                    }
                }
            }
        }
    }
}

