/*
 * serialReportTask.c
 *
 *  Created on: Jul 22, 2025
 *      Author: krzysiek
 */

#include "serialReportTask.h"
#include "guiTask.h"
#include "test.h"
#include "lvgl_port_display.h"


#include <stdbool.h>
#include <stdio.h>

typedef enum {
  TestDone_GPIO   = 0,
  TestDone_USB1   = 1,
  TestDone_USB2   = 2,
  TestDone_SDCARD = 3,
  TestDone_CAN    = 4,
  TestDone_RS485  = 5,
  TestDone_Button = 6,
  TestDone_All    = 7,
} TestDone_t;

typedef struct {
  TestType_t testType;
  bool testPassed;
  bool testFinished;
} TestState_t;

static TestState_t testStates[TestDone_All + 1];

osThreadId_t serialReportTaskHandle;

const osThreadAttr_t serialReportTask_attributes = {
  .name = "serialReportTask",
  .stack_size = 1280 * 4,
  .priority = (osPriority_t) osPriorityNormal,
};

#define KNRM  "\x1B[0m"
#define KRED  "\x1B[31m"
#define KGRN  "\x1B[32m"
#define KBLU  "\x1B[34m"

extern DSI_HandleTypeDef hdsi;

static void printInfo(char* info) {
  printf(KBLU"%s"KNRM"\r\n", info);
}

static void printResult(char* name, bool result) {
  printf("%s", name);
  if(result)
    printf(KGRN"OK"KNRM"\r\n");
  else
    printf(KRED"FAILED"KNRM"\r\n");
}

static void updateTestState(uint32_t testType, uint32_t result) {
  TestDone_t test;
  switch(testType) {
  case TestType_Gpio:
    test = TestDone_GPIO;
    break;
  case TestType_Usb1:
    test = TestDone_USB1;
    break;
  case TestType_Usb2:
    test = TestDone_USB2;
    break;
  case TestType_Sd:
    test = TestDone_SDCARD;
    break;
  case TestType_Can:
    test = TestDone_CAN;
    break;
  case TestType_Rs485:
    test = TestDone_RS485;
    break;
  case TestType_Button:
    test = TestDone_Button;
    break;
  default:
    return;
  }

  testStates[test].testType = testType;

  switch(result) {
  case TestResult_OK:
    testStates[test].testPassed = true;
    testStates[test].testFinished = true;
    break;
  case TestResult_Failed:
    testStates[test].testPassed = false;
    testStates[test].testFinished = true;
    break;
  default:
    testStates[test].testFinished = false;
    break;
  }

  for(int i = 0; i < TestDone_All; i++) {
    if(!testStates[i].testFinished)
      return;
  }

  testStates[TestDone_All].testFinished = true;
  testStates[TestDone_All].testPassed = true;

  for(int i = 0; i < TestDone_All; i++) {
    if(!testStates[i].testPassed) {
      testStates[TestDone_All].testPassed = false;
      return;
    }
  }
}

void serialReportTask(void *argument) {
  HAL_DSI_Start(&hdsi);

  HAL_DSI_ConfigFlowControl(&hdsi, DSI_FLOW_CONTROL_ALL);

  if(RVT70HSMNWC00_Init() != HAL_OK) {
      Error_Handler();
  }

  HAL_DSI_PatternGeneratorStart(&hdsi, 0, 0);

  vTaskDelay(3000 / portTICK_PERIOD_MS);
  printInfo("Press the [H.5], [H.4], [H.3] and [H.2] buttons");
  while(1) {
    GuiEvent_t event;
    if(osMessageQueueGet(guiEventQueueHandle, &event, NULL, 4000 / portTICK_PERIOD_MS) == osOK) {
      switch(event.type) {
      case GuiEventType_Test:
        updateTestState(event.data[0], event.data[1]);
        break;
      case GuiEventType_Button:
        if(event.data[0] == 0 && event.data[1] == 1)
          printResult("BUTTON [H.5] ", true);
        else if(event.data[0] == 1 && event.data[1] == 1)
          printResult("BUTTON [H.4] ", true);
        else if(event.data[0] == 2 && event.data[1] == 1)
          printResult("BUTTON [H.3] ", true);
        else if(event.data[0] == 3 && event.data[1] == 1)
          printResult("BUTTON [H.2] ", true);
        break;
      default:
        break;
      }

      if(!testStates[TestDone_All].testFinished)
        continue;
    }

    for(int i = 0; i < TestDone_All; i++)
      printResult(getTestTypeStr(testStates[i].testType), testStates[i].testPassed);
     printResult("TEST RESULT ", testStates[TestDone_All].testPassed);
     osThreadExit();
  }
}
