/*
 * rs485Task.c
 *
 *  Created on: Jan 27, 2025
 *      Author: krzysiek
 */

#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include "rs485Task.h"
#include "guiTask.h"
#include "test.h"

#define PING_DATA "12 34 56\n"

osThreadId_t rs485TaskHandle;

const osThreadAttr_t rs485Task_attributes = {
  .name = "rs485Task",
  .stack_size = configMINIMAL_STACK_SIZE * 4,
  .priority = (osPriority_t) osPriorityLow,
};

osMessageQueueId_t rs485QueueHandle;
const osMessageQueueAttr_t rs485Queue_attributes = {
  .name = "rs485Queue"
};

extern UART_HandleTypeDef huart6;

static uint8_t rxData[2 * sizeof(PING_DATA)];
static volatile uint8_t rxCompleted;

static void UART6_RxCpltCallback(UART_HandleTypeDef *huart)
{
  if(huart == &huart6) {
    rxCompleted = 1;
  }
}

void rs485Task(void *argument)
{
  bool testReported = false;
  huart6.RxCpltCallback = UART6_RxCpltCallback;

  while(1) {
    uint8_t trigger = 0;
    if(osMessageQueueGet(rs485QueueHandle, &trigger, NULL, 0) != osOK)
      continue;
    if(!trigger)
      continue;

    rxCompleted = 0;
    memset((void*)rxData, 0, sizeof(rxData));

    HAL_UART_Receive_IT(&huart6, rxData, 2*sizeof(PING_DATA));

    HAL_GPIO_WritePin(RS485_TX_EN_GPIO_Port, RS485_TX_EN_Pin, GPIO_PIN_SET);
    vTaskDelay(20 / portTICK_PERIOD_MS);
    HAL_UART_Transmit(&huart6, (uint8_t*)PING_DATA, sizeof(PING_DATA), 100);
    vTaskDelay(20 / portTICK_PERIOD_MS);
    HAL_GPIO_WritePin(RS485_TX_EN_GPIO_Port, RS485_TX_EN_Pin, GPIO_PIN_RESET);

    TickType_t timeout = 2000;
    TimeOut_t timeoutStart;
    vTaskSetTimeOutState(&timeoutStart);

    while(rxCompleted == 0) {
      vTaskDelay(20 / portTICK_PERIOD_MS);

      if(xTaskCheckForTimeOut(&timeoutStart, &timeout) != pdFALSE) {
        break;
      }
    }

    GuiEvent_t event;
    event.type = GuiEventType_Rs485;
    event.data[0] = 0;

    // Check ping data twice because of the local RS485 echo
    if((strncmp(PING_DATA, (char*)rxData, strlen(PING_DATA)) == 0)
      &&  (strncmp(PING_DATA, (char*)&rxData[sizeof(PING_DATA)], strlen(PING_DATA)) == 0)) {
      event.data[0] = 1;
    } else {
      printf("RS485 ERROR: [");
      for(int i=0; i<2 * sizeof(PING_DATA); i++)
        printf(" %x", rxData[i]);
      printf("]\r\n");
    }

    osMessageQueuePut(guiEventQueueHandle, &event, 0, 0);

    if(!testReported) {
      testReported = true;
      GuiEvent_t testReport;
      testReport.type = GuiEventType_Test;
      testReport.data[0] = TestType_Rs485;

      if(event.data[0] == 0)
        testReport.data[1] = TestResult_Failed;
      else
        testReport.data[1] = TestResult_OK;

      osMessageQueuePut(guiEventQueueHandle, &testReport, 0, 0);
    }
  }
}
