/*
 * ethTest.c
 *
 *  Created on: Jan 31, 2025
 *      Author: krzysiek
 */

#include <stdio.h>
#include <stdbool.h>
#include "ethTask.h"
#include "guiTask.h"
#include "test.h"


#include "lwip.h"
#include "ping.h"

osThreadId_t ethTaskHandle;


int pingTxCount;
int pingRxCount;
int pingMs;
int pingTimeoutCount;

const osThreadAttr_t ethTask_attributes = {
  .name = "ethTask",
  .stack_size = 1024 * 4,
  .priority = (osPriority_t) osPriorityLow,
};

osMessageQueueId_t ethQueueHandle;
const osMessageQueueAttr_t ethQueue_attributes = {
  .name = "ethQueue"
};

static uint32_t ethWaitForLink(struct netif* netif, uint32_t timeout_ms)
{
  timeout_ms /= 100;

  do {
    osDelay(100);
    if(netif_is_up(netif)) {
      break;
    }
  } while(--timeout_ms);

  if(netif_is_up(netif)) {
    extern ETH_HandleTypeDef heth;
    ETH_MACConfigTypeDef MACConf = {0};
    HAL_ETH_GetMACConfig(&heth, &MACConf);
    printf("ETH link UP %s/%s\r\n", MACConf.Speed ? "100M" : "10M", MACConf.DuplexMode ? "FD" : "HD");
  }

  return netif_is_up(netif);
}

static uint32_t  ethWaitForDhcp(struct netif* netif, uint32_t timeout_ms)
{
  uint32_t dhcp_status;

  timeout_ms /= 100;

  do {
    dhcp_status = dhcp_supplied_address(netif);

    if(dhcp_status) {

      struct dhcp *dhcp = netif_dhcp_data(netif);
      printf("DHCP IP address: %s\r\n", ip4addr_ntoa(&dhcp->offered_ip_addr));
      printf("DHCP Subnet mask: %s\r\n", ip4addr_ntoa(&dhcp->offered_sn_mask));
      printf("DHCP Default gateway: %s\r\n", ip4addr_ntoa(&dhcp->offered_gw_addr));

      break;
    }
    osDelay(100);

  } while(--timeout_ms);
  return dhcp_status;
}

void ethTask(void *pvParameters) {
    MX_LWIP_Init();

    struct netif* netif = netif_find("st");

    if(!ethWaitForLink(netif, 5000))
      ; //TODO break test

    if(!ethWaitForDhcp(netif, 5000))
      ;  //TODO break test

    bool testReported = false;
    ip4_addr_t netif_pingaddr;
    IP4_ADDR(&netif_pingaddr, 8, 8, 8, 8);

    osThreadId_t pingHandle = ping_init(&netif_pingaddr);

    while(1) {

        uint8_t trigger = 0;
        if(osMessageQueueGet(ethQueueHandle, &trigger, NULL, osWaitForever) != osOK)
            continue;
        if(!trigger)
            continue;

        pingTxCount = 0;
        pingRxCount = 0;
        pingTimeoutCount = 0;
        pingMs = 0;

        osThreadFlagsSet(pingHandle, 1);

        while(!pingRxCount && !pingTimeoutCount) {
            vTaskDelay(100 / portTICK_PERIOD_MS);
        }

        if(pingRxCount == 1 && pingTimeoutCount == 0) {
            GuiEvent_t event;
            event.type = GuiEventType_Eth;
            event.data[0] = 1;
            event.data[1] = pingMs;

            osMessageQueuePut(guiEventQueueHandle, &event, 0, 0);

            if(!(testReported)) {
                testReported = true;
                GuiEvent_t testReport;
                testReport.type = GuiEventType_Test;
                testReport.data[0] = TestType_Eth;
                testReport.data[1] = TestResult_OK;
                osMessageQueuePut(guiEventQueueHandle, &testReport, 0, 0);
            }

        } else {
            GuiEvent_t event;
            event.type = GuiEventType_Eth;
            event.data[0] = 0;
            event.data[1] = pingMs;
            osMessageQueuePut(guiEventQueueHandle, &event, 0, 0);

            if(!(testReported)) {
                testReported = true;
                GuiEvent_t testReport;
                testReport.type = GuiEventType_Test;
                testReport.data[0] = TestType_Eth;
                testReport.data[1] = TestResult_Failed;
                osMessageQueuePut(guiEventQueueHandle, &testReport, 0, 0);
            }
        }
    }
}
