/*
 * canTask.c
 *
 *  Created on: Jan 27, 2025
 *      Author: krzysiek
 */

#include <stdbool.h>
#include <string.h>
#include "canTask.h"
#include "guiTask.h"
#include "test.h"

osThreadId_t canTaskHandle;

const osThreadAttr_t canTask_attributes = {
  .name = "canTask",
  .stack_size = configMINIMAL_STACK_SIZE * 4,
  .priority = (osPriority_t) osPriorityLow,
};

osMessageQueueId_t canQueueHandle;
const osMessageQueueAttr_t canQueue_attributes = {
  .name = "canQueue"
};

extern FDCAN_HandleTypeDef hfdcan1;
static volatile uint32_t rxCompleted = 0;
static FDCAN_RxHeaderTypeDef rxHeader;
static uint8_t rxBuffer[8];

void HAL_FDCAN_RxFifo0Callback(FDCAN_HandleTypeDef *hfdcan, uint32_t RxFifo0ITs)
{
  if((RxFifo0ITs & FDCAN_IT_RX_FIFO0_NEW_MESSAGE) != RESET)
  {
    /* Retrieve Rx messages from RX FIFO0 */
    if (HAL_FDCAN_GetRxMessage(hfdcan, FDCAN_RX_FIFO0, &rxHeader, rxBuffer) != HAL_OK)
      Error_Handler();

    rxCompleted = 1;
    if (HAL_FDCAN_ActivateNotification(hfdcan, FDCAN_IT_RX_FIFO0_NEW_MESSAGE, 0) != HAL_OK)
      Error_Handler();
  }
}

static void sendPingResult(uint8_t success, bool* testReported) {
    GuiEvent_t event;
    event.type = GuiEventType_Can;
    event.data[0] = success;

    osMessageQueuePut(guiEventQueueHandle, &event, 0, 0);

    if(!(*testReported)) {
        *testReported = true;
        GuiEvent_t testReport;

        testReport.type = GuiEventType_Test;
        testReport.data[0] = TestType_Can;
        if(success)
            testReport.data[1] = TestResult_OK;
        else
            testReport.data[1] = TestResult_Failed;
        osMessageQueuePut(guiEventQueueHandle, &testReport, 0, 0);    }
}

void canTask(void *argument)
{
  bool testReported = false;
  hfdcan1.RxFifo0Callback = HAL_FDCAN_RxFifo0Callback;

  FDCAN_FilterTypeDef sFilterConfig;

  sFilterConfig.IdType = FDCAN_STANDARD_ID;
  sFilterConfig.FilterIndex = 0;
  sFilterConfig.FilterType = FDCAN_FILTER_MASK;
  sFilterConfig.FilterConfig = FDCAN_FILTER_TO_RXFIFO0;
  sFilterConfig.FilterID1 = 0;
  sFilterConfig.FilterID2 = 0xFFFFFFFF;
  sFilterConfig.RxBufferIndex = 0;

  if (HAL_FDCAN_ConfigFilter(&hfdcan1, &sFilterConfig) != HAL_OK)
    Error_Handler();

  if(HAL_FDCAN_Start(&hfdcan1)!= HAL_OK)
    Error_Handler();

  if (HAL_FDCAN_ActivateNotification(&hfdcan1, FDCAN_IT_RX_FIFO0_NEW_MESSAGE, 0) != HAL_OK)
    Error_Handler();

  while(1) {
    uint8_t trigger = 0;
    if(osMessageQueueGet(canQueueHandle, &trigger, NULL, 0) != osOK)
      continue;
    if(!trigger)
      continue;

    rxCompleted = 0;
    uint8_t txData[] = {0x12, 0x34, 0x56};
    memset(rxBuffer, 0, sizeof(rxBuffer));

    FDCAN_TxHeaderTypeDef   txHeader;
    txHeader.Identifier = 0x123;
    txHeader.IdType = FDCAN_STANDARD_ID;
    txHeader.TxFrameType = FDCAN_DATA_FRAME;
    txHeader.DataLength = sizeof(txData);
    txHeader.ErrorStateIndicator = FDCAN_ESI_ACTIVE;
    txHeader.BitRateSwitch = FDCAN_BRS_OFF;
    txHeader.FDFormat = FDCAN_CLASSIC_CAN;
    txHeader.TxEventFifoControl = FDCAN_NO_TX_EVENTS;
    txHeader.MessageMarker = 0;

    HAL_FDCAN_AddMessageToTxFifoQ(&hfdcan1, &txHeader, txData);

    TickType_t timeout = 2000;
    TimeOut_t timeoutStart;
    bool isTimeout = false;
    vTaskSetTimeOutState(&timeoutStart);

    while(rxCompleted == 0) {
      vTaskDelay(20 / portTICK_PERIOD_MS);

      if(xTaskCheckForTimeOut(&timeoutStart, &timeout) != pdFALSE) {
        isTimeout = true;
        break;
      }
    }

    if(isTimeout) {
      sendPingResult(0, &testReported);
    } else if(rxHeader.Identifier == 0x123 && rxBuffer[0] == 0x12 &&
              rxBuffer[1] == 0x34 && rxBuffer[2] == 0x56) {
            sendPingResult(1, &testReported);
    } else {
            sendPingResult(0, &testReported);
    }
  }
}
