/*
 * audioTask.c
 *
 *  Created on: Jan 30, 2025
 *      Author: krzysiek
 */

#include <stdio.h>
#include "audioTask.h"

#define CODEC_I2C_ADDR  0x1A
#define CODEC_SLAVE_ADDR  (CODEC_I2C_ADDR << 1)
#define AUDIO_BUFFER_SIZE   (uint16_t)0x4000

extern I2C_HandleTypeDef hi2c1;
extern I2S_HandleTypeDef hi2s2;

osThreadId_t audioTaskHandle;

const osThreadAttr_t audioTask_attributes = {
  .name = "audioTask",
  .stack_size = 256 * 4,
  .priority = (osPriority_t) osPriorityRealtime,
};

#if IS_MEM_MODULE == 1
static uint16_t* audioBuffer1 = (uint16_t*)0xC0C00000; //[AUDIO_BUFFER_SIZE];
static uint16_t* audioBuffer2 = (uint16_t*)0xC0C04000; //[AUDIO_BUFFER_SIZE];
#else
static uint16_t audioBuffer1[AUDIO_BUFFER_SIZE];
static uint16_t audioBuffer2[AUDIO_BUFFER_SIZE];
#endif

static void audioXferDone(I2S_HandleTypeDef *hi2s) {
  osThreadFlagsSet(audioTaskHandle, 1);
}

static HAL_StatusTypeDef NAU88C22_WriteReg(uint8_t reg, uint16_t val)
{
  uint8_t xfer[2];

  xfer[0] = (reg << 1) | ((val >> 8) & 0x01);
  xfer[1] = val & 0xFF;

  return HAL_I2C_Master_Transmit(&hi2c1, CODEC_SLAVE_ADDR, xfer, sizeof(xfer), 10);
}

static HAL_StatusTypeDef NAU88C22_ReadReg(uint8_t reg, uint16_t *val)
{
  HAL_StatusTypeDef result;
  uint8_t readValue[2] = {0, 0};

  reg <<= 1;

  result = HAL_I2C_Mem_Read(&hi2c1, CODEC_SLAVE_ADDR, reg, sizeof(reg), readValue, sizeof(readValue), 10);

  *val = (readValue[0] << 8) | readValue[1];

  return result;
}

void audioTask(void *pvParameters) {
  // software reset
  NAU88C22_WriteReg(0x00, 0x0001);
  osDelay(5);
  NAU88C22_WriteReg(0x01, 0x0000);
  osDelay(5);

  uint16_t codecRev = 0;
  uint16_t codecId = 0;
  NAU88C22_ReadReg(0x3E, &codecRev);
  NAU88C22_ReadReg(0x3F, &codecId);
  printf("NAU88C22 codec ID: %02x, rev: %02x\r\n", codecId, codecRev);
  if(codecId != 0x1A || codecRev != 0x7F) {
    printf("NAU88C22 codec ID ERROR\r\n");
    while(1)
      osDelay(1000);
  }

  /* enable used modules */
  NAU88C22_WriteReg(0x01, 0x001F);
  NAU88C22_WriteReg(0x02, 0x01FF);
  NAU88C22_WriteReg(0x03, 0x01ef);

  /* clock source is MCLK, divide by 1, FS and BCLK are inputs */
  NAU88C22_WriteReg(0x06, 0x0000);

  NAU88C22_WriteReg(0x04, 0x0010);// I2S protocol, 16-bit

  NAU88C22_WriteReg(0x34, 0x0030); // Set LHP/RHP volume
  NAU88C22_WriteReg(0x35, 0x0130);

  hi2s2.TxRxCpltCallback = audioXferDone;

  HAL_I2SEx_TransmitReceive_DMA(&hi2s2, audioBuffer1, audioBuffer2, AUDIO_BUFFER_SIZE);

  uint8_t bufferSwitch = 1;

  for(;;)
  {
    osThreadFlagsWait (1, osFlagsWaitAny, osWaitForever);
    HAL_I2S_DMAStop(&hi2s2);
    if(bufferSwitch == 1) {
      HAL_I2SEx_TransmitReceive_DMA(&hi2s2, audioBuffer2, audioBuffer1, AUDIO_BUFFER_SIZE);
      bufferSwitch = 0;
    } else {
      HAL_I2SEx_TransmitReceive_DMA(&hi2s2, audioBuffer1, audioBuffer2, AUDIO_BUFFER_SIZE);
      bufferSwitch = 1;
    }
  }
}
