/*********************
 *      INCLUDES
 *********************/

#include "lvgl_port_display.h"
#include "cmsis_os.h"

extern DMA2D_HandleTypeDef hdma2d;
extern LTDC_HandleTypeDef hltdc;
extern DSI_HandleTypeDef hdsi;

#define DIRECT_MODE

/**********************
 *  STATIC PROTOTYPES
 **********************/

static void disp_flush (lv_display_t *, const lv_area_t *, uint8_t *);

#ifndef DIRECT_MODE
static void disp_flush_complete (DMA2D_HandleTypeDef*);
#endif

/**********************
 *  STATIC VARIABLES
 **********************/
static lv_display_t * disp;

#ifdef DIRECT_MODE
#define LVGL_BUFFER_1_ADDR_AT_SDRAM	(0xC0000000)
#define LVGL_BUFFER_2_ADDR_AT_SDRAM	(0xC0400000)
#else
#define LVGL_BUFFER_1_ADDR_AT_SDRAM	(0xD01F4000)
#define LVGL_BUFFER_2_ADDR_AT_SDRAM	(0xD03E8000)
#endif

/**********************
 *   GLOBAL FUNCTIONS
 **********************/

void lvgl_display_init (void)
{
	/* display initialization */

    HAL_DSI_Start(&hdsi);

    /* Enable the DSI BTW for read operations */
    HAL_DSI_ConfigFlowControl(&hdsi, DSI_FLOW_CONTROL_ALL);

    if(RVT70HSMNWC00_Init() != HAL_OK) {
        Error_Handler();
    }

	disp = lv_display_create(MY_DISP_HOR_RES, MY_DISP_VER_RES);
#ifdef DIRECT_MODE
	lv_display_set_buffers(disp, (void*) LVGL_BUFFER_1_ADDR_AT_SDRAM, (void*) LVGL_BUFFER_2_ADDR_AT_SDRAM, MY_DISP_HOR_RES * MY_DISP_VER_RES * 2, LV_DISPLAY_RENDER_MODE_DIRECT);
	HAL_LTDC_SetAddress(&hltdc, (uint32_t)LVGL_BUFFER_2_ADDR_AT_SDRAM, 0);	// start with the second buffer: LVGL will render into the first buffer
#else
	__attribute__((section(".my_ram_d2_data"))) static uint8_t buf1[288*1024];

	lv_display_set_buffers(disp, (void*) buf1, NULL, sizeof(buf1), LV_DISPLAY_RENDER_MODE_PARTIAL);

	/* interrupt callback for DMA2D transfer */
	hdma2d.XferCpltCallback = disp_flush_complete;
//	lv_display_set_flush_wait_cb(disp, disp_flush_wait);
//	lv_thread_sync_init(&sync);
#endif
	lv_display_set_flush_cb(disp, disp_flush);
}

void lvgl_display_backlight_enable(uint8_t state)
{
  HAL_GPIO_WritePin(DISP_BACKLIGHT_EN_GPIO_Port, DISP_BACKLIGHT_EN_Pin, state ? GPIO_PIN_SET : GPIO_PIN_RESET);
}

void lvgl_display_backlight_change(void)
{
  HAL_GPIO_TogglePin(DISP_BACKLIGHT_EN_GPIO_Port, DISP_BACKLIGHT_EN_Pin);
}

void lvgl_display_brightness_enable(uint8_t state)
{
  HAL_GPIO_WritePin(DISP_PWM_GPIO_Port, DISP_PWM_Pin, state ? GPIO_PIN_SET : GPIO_PIN_RESET);
}

void lvgl_display_brightness_change(void)
{
  HAL_GPIO_TogglePin(DISP_PWM_GPIO_Port, DISP_PWM_Pin);
}

/**********************
 *   STATIC FUNCTIONS
 **********************/

static void disp_flush (lv_display_t * display,
            const lv_area_t * area,
            uint8_t * px_map)
{
#ifdef DIRECT_MODE
	if (lv_display_flush_is_last(disp)) {
		SCB_CleanInvalidateDCache();
		// wait for VSYNC to avoid tearing
		while (!(LTDC->CDSR & LTDC_CDSR_VSYNCS));
		// swap framebuffers (NOTE: LVGL will swap the buffers in the background, so here we can set the LCD framebuffer to the current LVGL buffer, which has been just completed)
		HAL_LTDC_SetAddress(&hltdc, (uint32_t)(lv_display_get_buf_active(disp)->data), 0);
	}
	lv_display_flush_ready(disp);
#else
  lv_coord_t width = lv_area_get_width(area);
  lv_coord_t height = lv_area_get_height(area);

  SCB_CleanInvalidateDCache();

  DMA2D->CR = 0x0U << DMA2D_CR_MODE_Pos;
  DMA2D->FGPFCCR = DMA2D_INPUT_RGB565;
  DMA2D->FGMAR = (uint32_t)px_map;
  DMA2D->FGOR = 0;
  DMA2D->OPFCCR = DMA2D_OUTPUT_RGB565;
  DMA2D->OMAR = hltdc.LayerCfg[0].FBStartAdress + 2 * \
                (area->y1 * MY_DISP_HOR_RES + area->x1);
  DMA2D->OOR = MY_DISP_HOR_RES - width;
  DMA2D->NLR = (width << DMA2D_NLR_PL_Pos) | (height << DMA2D_NLR_NL_Pos);
  DMA2D->IFCR = 0x3FU;
  DMA2D->CR |= DMA2D_CR_TCIE;
  DMA2D->CR |= DMA2D_CR_START;
#endif
}

HAL_StatusTypeDef RVT70HSMNWC00_Init(void)
{
    uint32_t i;
    HAL_StatusTypeDef status = 0;

    static const uint8_t displayInitSequence[][2] = {
//        {0x87, 0x5A},
//        {0xB0, 0x80},
        {0xB2, 0x50},   // 2-lane interface
        {0x80, 0x4B},
        {0x81, 0xFF},
        {0x82, 0x1A},
        {0x83, 0x88},
        {0x84, 0x8F},
        {0x85, 0x35},
        {0x86, 0xB0},
    };

    /* Perform reset. */
    HAL_GPIO_WritePin(DISP_DSI_RESET_GPIO_Port, DISP_DSI_RESET_Pin, GPIO_PIN_RESET);
    osDelay(1);
    HAL_GPIO_WritePin(DISP_DSI_RESET_GPIO_Port, DISP_DSI_RESET_Pin, GPIO_PIN_SET);
    osDelay(5);

    status = HAL_DSI_ShortWrite(&hdsi, 0, DSI_DCS_SHORT_PKT_WRITE_P0, 0x01, 0);
    if (HAL_OK != status) {
        return status;
    }
    osDelay(120);

    /* Set the LCM init settings. */
    for (i = 0; i < sizeof(displayInitSequence)/sizeof(displayInitSequence[0]); i++) {
        status = HAL_DSI_ShortWrite(&hdsi, 0, DSI_DCS_SHORT_PKT_WRITE_P1, displayInitSequence[i][0], displayInitSequence[i][1]);
        if (HAL_OK != status) {
            return status;
        }
    }
    osDelay(50);

    /* Exit sleep mode */
    status = HAL_DSI_ShortWrite(&hdsi, 0, DSI_DCS_SHORT_PKT_WRITE_P0, 0x11, 0);
    if (HAL_OK != status) {
        return status;
    }
    osDelay(120);

    /* Set display on. */
    status = HAL_DSI_ShortWrite(&hdsi, 0, DSI_DCS_SHORT_PKT_WRITE_P0, 0x29, 0);
    if (HAL_OK != status)
    {
        return status;
    }
    osDelay(120);

    return HAL_OK;
}

#ifndef DIRECT_MODE

static void
disp_flush_complete (DMA2D_HandleTypeDef *hdma2d)
{
//	lv_thread_sync_signal_isr(&sync);

  lv_display_flush_ready(disp);
}

#endif
