/*
 * gui.c
 *
 *  Created on: Jan 29, 2025
 *      Author: krzysiek
 */

#include "cmsis_os.h"
#include "ledTask.h"
#include "ethTask.h"
#include "canTask.h"
#include "rs485Task.h"

#include "lvgl/lvgl.h"
#include "lvgl_port_display.h"
#include "test.h"
#include "gui.h"

/*
#include <tasks/canTask.h>

#include "board.h"
*/

#define LEDS_COUNT 4
#define TABLE_CELL_GREEN LV_TABLE_CELL_CTRL_CUSTOM_1
#define TABLE_CELL_RED LV_TABLE_CELL_CTRL_CUSTOM_2
#define TABLE_CELL_BLACK LV_TABLE_CELL_CTRL_CUSTOM_3
#define TABLE_CELL_GREY LV_TABLE_CELL_CTRL_CUSTOM_4

static lv_obj_t * table;
static lv_style_t styleMonospace;

typedef struct {
    uint8_t ledId;
    uint8_t stateId;
} LedData_t;

static LedData_t led1Data = {.ledId = 0};
static LedData_t led2Data = {.ledId = 1};
static LedData_t led3Data = {.ledId = 2};
static LedData_t led4Data = {.ledId = 3};
static LedData_t led5Data = {.ledId = 4};

static lv_obj_t* leds[LEDS_COUNT];

__attribute__((aligned(64)))
uint8_t imageBuffer[IMAGE_BUFFER_HEIGHT * IMAGE_BUFFER_WIDTH * IMAGE_BUFFER_BPP];

static const lv_img_dsc_t imageData = {
    .header.w = IMAGE_BUFFER_WIDTH,
    .header.h = IMAGE_BUFFER_HEIGHT,
    .header.cf = LV_COLOR_FORMAT_NATIVE,
    .data = imageBuffer,
    .data_size = sizeof(imageBuffer),
};

extern const lv_img_dsc_t somlabs_logo;

static lv_obj_t* cameraImage;

static lv_obj_t* usb1VidLabel;
static lv_obj_t* usb1PidLabel;
static lv_obj_t* usb2VidLabel;
static lv_obj_t* usb2PidLabel;

static lv_obj_t* pingRs485Label;
static uint32_t pingRs485Counter;

static lv_obj_t* pingCanLabel;
static uint32_t pingCanCounter;

static lv_obj_t* pingEthLabel;
static uint32_t pingEthCounter;

static void backlightEnableBtnEventCb(lv_event_t* e) {
    lv_event_code_t code = lv_event_get_code(e);
    if(code == LV_EVENT_CLICKED)
        lvgl_display_backlight_change();
}

static void backlightMinMaxBtnEventCb(lv_event_t * e) {
    lv_event_code_t code = lv_event_get_code(e);
    if(code == LV_EVENT_CLICKED)
        lvgl_display_brightness_change();
}

static void pingRs485BtnCb(lv_event_t * e) {
    lv_event_code_t code = lv_event_get_code(e);
    if(code == LV_EVENT_CLICKED) {
        lv_label_set_text(pingRs485Label, "PING: ...");
        uint8_t pingTrigger = 1;
        osMessageQueuePut(rs485QueueHandle, &pingTrigger, 0, 0);
    }
}

static void pingCanBtnCb(lv_event_t * e) {
    lv_event_code_t code = lv_event_get_code(e);
    if(code == LV_EVENT_CLICKED) {
        lv_label_set_text(pingCanLabel, "PING: ...");
        uint8_t pingTrigger = 1;
        osMessageQueuePut(canQueueHandle, &pingTrigger, 0, 0);
    }
}

static void pingEthBtnCb(lv_event_t * e) {
    lv_event_code_t code = lv_event_get_code(e);
    if(code == LV_EVENT_CLICKED) {
        lv_label_set_text(pingEthLabel, "PING: ...");
        uint8_t pingTrigger = 1;
        osMessageQueuePut(ethQueueHandle, &pingTrigger, 0, 0);
    }
}

static void tableDrawEventCb(lv_event_t * e) {
    lv_obj_t* obj = lv_event_get_target_obj(e);
    lv_draw_task_t* draw_task = lv_event_get_draw_task(e);
    lv_draw_dsc_base_t* base_dsc = draw_task->draw_dsc;

    if(base_dsc->part == LV_PART_ITEMS) {
        uint32_t row = base_dsc->id1;
        uint32_t col = base_dsc->id2;

        lv_draw_label_dsc_t* label_draw_dsc = lv_draw_task_get_label_dsc(draw_task);
        if(!label_draw_dsc)
            return;

        if(lv_table_has_cell_ctrl(obj, row, col, TABLE_CELL_GREEN))
          label_draw_dsc->color = lv_palette_main(LV_PALETTE_GREEN);
        else if(lv_table_has_cell_ctrl(obj, row, col, TABLE_CELL_RED))
          label_draw_dsc->color = lv_palette_main(LV_PALETTE_RED);
        else if(lv_table_has_cell_ctrl(obj, row, col, TABLE_CELL_GREY))
           label_draw_dsc->color = lv_palette_main(LV_PALETTE_GREY);
    }
}

static void checkBoxEventHandler(lv_event_t * e)
{
    LedData_t* ledData = lv_event_get_user_data(e);
    lv_obj_t* container = lv_event_get_current_target(e);
    lv_obj_t* currentButton = lv_event_get_target(e);
    lv_obj_t* oldButton = lv_obj_get_child(container, ledData->stateId);
    uint32_t newActiveId = lv_obj_get_index(currentButton);

    if(currentButton == container)
        return;

    if(newActiveId != ledData->stateId) {
        lv_obj_clear_state(oldButton, LV_STATE_CHECKED);
        ledData->stateId = newActiveId;
        LedEvent_t event;
        event.ledId = ledData->ledId;
        event.ledState = ledData->stateId - 1;
        osMessageQueuePut(ledQueueHandle, &event, 0, 0);
    } else {
        lv_obj_add_state(oldButton, LV_STATE_CHECKED);
    }
}

static void checkButtonCreate(lv_obj_t* parent, const char* label)
{
    lv_obj_t * obj = lv_checkbox_create(parent);
    lv_checkbox_set_text(obj, label);
    lv_obj_add_flag(obj, LV_OBJ_FLAG_EVENT_BUBBLE);
}

static void checkBoxCreate(lv_obj_t* parent, uint32_t x, uint32_t y, LedData_t* ledData, const char* title) {
    lv_obj_t* container = lv_obj_create(parent);
    lv_obj_set_flex_flow(container, LV_FLEX_FLOW_COLUMN);
    lv_obj_set_size(container, 140, 180);
    lv_obj_set_pos(container, x, y);
    lv_obj_add_event_cb(container, checkBoxEventHandler, LV_EVENT_CLICKED, ledData);

    lv_obj_t* label = lv_label_create(container);
    lv_label_set_text(label, title);
    lv_obj_center(label);

    checkButtonCreate(container, "Blink");
    checkButtonCreate(container, "On");
    checkButtonCreate(container, "Off");

    ledData->stateId = 1;
    lv_obj_add_state(lv_obj_get_child(container, 1), LV_STATE_CHECKED);
}

static void ledBoxCreate(lv_obj_t* parent, uint32_t x, uint32_t y) {
    lv_obj_t* container = lv_obj_create(parent);
    lv_obj_set_size(container, 300, 110);
    lv_obj_set_pos(container, x, y);

    lv_obj_t* label = lv_label_create(container);
    lv_label_set_text(label, "CB Buttons");
    lv_obj_set_pos(label, 70, 0);

    for(int i = 0; i < LEDS_COUNT; i++) {
        leds[i] = lv_led_create(container);
        lv_led_set_color(leds[i], lv_palette_main(LV_PALETTE_RED));
        lv_led_off(leds[i]);
        lv_obj_set_pos(leds[i], 25 + (i * 60), 35);
    }
}

static void initCbTestTab(lv_obj_t* parent) {
    lv_obj_t* backlightEnableBtn = lv_btn_create(parent);
    lv_obj_set_pos(backlightEnableBtn, 10, 20);
    lv_obj_set_size(backlightEnableBtn, 300, 100);
    lv_obj_add_event_cb(backlightEnableBtn, backlightEnableBtnEventCb, LV_EVENT_CLICKED, NULL);

    lv_obj_t* label = lv_label_create(backlightEnableBtn);
    lv_label_set_text(label, "Backlight ON/OFF");
    lv_obj_center(label);

    lv_obj_t* backlightMinMaxBtn = lv_btn_create(parent);
    lv_obj_set_pos(backlightMinMaxBtn, 10, 160);
    lv_obj_set_size(backlightMinMaxBtn, 300, 100);
    lv_obj_add_event_cb(backlightMinMaxBtn, backlightMinMaxBtnEventCb, LV_EVENT_CLICKED, NULL);

    label = lv_label_create(backlightMinMaxBtn);
    lv_label_set_text(label, "Backlight MIN/MAX");
    lv_obj_center(label);

    lv_style_init(&styleMonospace);
    lv_style_set_text_font(&styleMonospace, &lv_font_unscii_8);

    table = lv_table_create(parent);
    lv_obj_add_style(table, &styleMonospace, LV_PART_ITEMS);

    lv_table_set_col_cnt(table, 1);
    lv_table_set_row_cnt(table, TestType_TestsCnt);
    lv_table_set_col_width(table, 0, 250);
    lv_obj_set_size(table, 260, 510);
    lv_obj_set_pos(table, 720, 0);
    lv_obj_set_style_pad_ver(table, 17, LV_PART_ITEMS);
    lv_obj_add_event_cb(table, tableDrawEventCb, LV_EVENT_DRAW_TASK_ADDED, NULL);
    lv_obj_add_flag(table, LV_OBJ_FLAG_SEND_DRAW_TASK_EVENTS);
    for(int i = 0; i < TestType_TestsCnt; i++)
        guiSetTestTableResult((TestType_t)i, TestResult_InProgress);

    ledBoxCreate(parent, 10, 300);

    cameraImage = lv_img_create(parent);
    lv_img_set_src(cameraImage, &imageData);
    lv_obj_set_pos(cameraImage, 350, 25);
    lv_img_set_angle(cameraImage, 1800);
    lv_obj_t* logoImage = lv_img_create(parent);
    lv_img_set_src(logoImage, &somlabs_logo);
    lv_obj_set_pos(logoImage, 385, 320);
}

static void initLedTab(lv_obj_t* parent) {
    checkBoxCreate(parent, 20, 160, &led1Data, "LED A.0");
    checkBoxCreate(parent, 220, 160, &led2Data, "LED A.5");
    checkBoxCreate(parent, 420, 160, &led3Data, "LED I.11");
    checkBoxCreate(parent, 620, 160, &led4Data, "LED E.6");
    checkBoxCreate(parent, 820, 160, &led5Data, "LED E.5");
}

static void initUsbTab(lv_obj_t* parent) {
    lv_obj_t* label = lv_label_create(parent);
    lv_label_set_text(label, "USB Mass Storage Device detection");
    lv_obj_set_pos(label, 325, 25);

    lv_obj_t* container = lv_obj_create(parent);
    lv_obj_set_flex_flow(container, LV_FLEX_FLOW_COLUMN);
    lv_obj_set_flex_align(container, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND);
    lv_obj_set_size(container, 200, 200);
    lv_obj_set_pos(container, 210, 100);

    label = lv_label_create(container);
    lv_label_set_text(label, "USB1");
    lv_obj_center(label);

    usb1VidLabel = lv_label_create(container);
    lv_label_set_text(usb1VidLabel, "VID: ------");
    lv_obj_center(usb1VidLabel);

    usb1PidLabel = lv_label_create(container);
    lv_label_set_text(usb1PidLabel, "PID: ------");
    lv_obj_center(label);

    container = lv_obj_create(parent);
    lv_obj_set_flex_flow(container, LV_FLEX_FLOW_COLUMN);
    lv_obj_set_flex_align(container, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND);
    lv_obj_set_size(container, 200, 200);
    lv_obj_set_pos(container, 610, 100);

    label = lv_label_create(container);
    lv_label_set_text(label, "USB2");
    lv_obj_center(label);

    usb2VidLabel = lv_label_create(container);
    lv_label_set_text(usb2VidLabel, "VID: ------");
    lv_obj_center(label);

    usb2PidLabel = lv_label_create(container);
    lv_label_set_text(usb2PidLabel, "PID: ------");
    lv_obj_center(label);
}

static void initPingTab(lv_obj_t* parent) {
    lv_obj_t* label = lv_label_create(parent);
    lv_label_set_text(label, "Ping RS485 / CAN / ETH interfaces");
    lv_obj_set_pos(label, 325, 25);

    lv_obj_t* pingRs485Btn = lv_btn_create(parent);
    lv_obj_set_size(pingRs485Btn, 250, 100);
    lv_obj_set_pos(pingRs485Btn, 135, 100);
    lv_obj_add_event_cb(pingRs485Btn, pingRs485BtnCb, LV_EVENT_CLICKED, NULL);

    label = lv_label_create(pingRs485Btn);
    lv_label_set_text(label, "Ping");
    lv_obj_center(label);

    lv_obj_t* container = lv_obj_create(parent);
    lv_obj_set_flex_flow(container, LV_FLEX_FLOW_COLUMN);
    lv_obj_set_flex_align(container, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND);
    lv_obj_set_size(container, 250, 200);
    lv_obj_set_pos(container, 135, 250);

    label = lv_label_create(container);
    lv_label_set_text(label, "RS485");
    lv_obj_center(label);

    pingRs485Counter = 0;
    pingRs485Label = lv_label_create(container);
    lv_label_set_text(pingRs485Label, "PING: 0");
    lv_obj_center(pingRs485Label);

    lv_obj_t* pingCanBtn = lv_btn_create(parent);
    lv_obj_set_size(pingCanBtn, 250, 100);
    lv_obj_set_pos(pingCanBtn, 400, 100);
    lv_obj_add_event_cb(pingCanBtn, pingCanBtnCb, LV_EVENT_CLICKED, NULL);

    label = lv_label_create(pingCanBtn);
    lv_label_set_text(label, "Ping");
    lv_obj_center(label);

    container = lv_obj_create(parent);
    lv_obj_set_flex_flow(container, LV_FLEX_FLOW_COLUMN);
    lv_obj_set_flex_align(container, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND);
    lv_obj_set_size(container, 250, 200);
    lv_obj_set_pos(container, 400, 250);

    label = lv_label_create(container);
    lv_label_set_text(label, "CAN");
    lv_obj_center(label);

    pingCanCounter = 0;
    pingCanLabel = lv_label_create(container);
    lv_label_set_text(pingCanLabel, "PING: 0");
    lv_obj_center(pingCanLabel);

    lv_obj_t* pingEthBtn = lv_btn_create(parent);
    lv_obj_set_size(pingEthBtn, 250, 100);
    lv_obj_set_pos(pingEthBtn, 675, 100);
    lv_obj_add_event_cb(pingEthBtn, pingEthBtnCb, LV_EVENT_CLICKED, NULL);

    label = lv_label_create(pingEthBtn);
    lv_label_set_text(label, "Ping");
    lv_obj_center(label);

    container = lv_obj_create(parent);
    lv_obj_set_flex_flow(container, LV_FLEX_FLOW_COLUMN);
    lv_obj_set_flex_align(container, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND, LV_FLEX_ALIGN_SPACE_AROUND);
    lv_obj_set_size(container, 250, 200);
    lv_obj_set_pos(container, 675, 250);

    label = lv_label_create(container);
    lv_label_set_text(label, "ETH");
    lv_obj_center(label);

    pingEthCounter = 0;
    pingEthLabel = lv_label_create(container);
    lv_label_set_text(pingEthLabel, "PING: 0");
    lv_obj_center(pingEthLabel);
}
void guiInit(void) {
    lv_obj_t* tabview = lv_tabview_create(lv_scr_act());
    lv_tabview_set_tab_bar_position(tabview, LV_DIR_TOP);
    lv_tabview_set_tab_bar_size(tabview, 50);

    lv_obj_t* cbTestTab = lv_tabview_add_tab(tabview, "CB Test");
    lv_obj_t* ledTab = lv_tabview_add_tab(tabview, "LEDs");
    lv_obj_t* usbTab = lv_tabview_add_tab(tabview, "USB");
    lv_obj_t* pingTab = lv_tabview_add_tab(tabview, "Ping");

    initCbTestTab(cbTestTab);
    initLedTab(ledTab);
    initUsbTab(usbTab);
    initPingTab(pingTab);
}

void guiSetTestTableResult(TestType_t type, TestResult_t result) {
    lv_table_set_cell_value_fmt(table, type, 0, "%s%s", getTestTypeStr(type), getTestResultStr(result));
    switch(result) {
    case TestResult_OK:
        lv_table_add_cell_ctrl(table, type, 0, TABLE_CELL_GREEN);
        break;
    case TestResult_Failed:
        lv_table_add_cell_ctrl(table, type, 0, TABLE_CELL_RED);
        break;
    case TestResult_NotAvailable:
        lv_table_add_cell_ctrl(table, type, 0, TABLE_CELL_GREY);
        break;
    default:
        lv_table_add_cell_ctrl(table, type, 0, TABLE_CELL_BLACK);
        break;
    }
}

void guiSetLedState(uint8_t id, uint8_t state) {
    if(id > LEDS_COUNT)
        return;

    if(state == 0)
        lv_led_off(leds[id]);
    else
        lv_led_set_brightness(leds[id], 255);
}

void guiUpdateCameraImage(void) {
    lv_obj_invalidate(cameraImage);
}

void guiSetUsbLabel(uint8_t usbId, uint8_t attached, uint32_t vid, uint32_t pid) {
    if(usbId == 0) {
        if(attached) {
            lv_label_set_text_fmt(usb1VidLabel, "VID: 0x%lx", vid);
            lv_label_set_text_fmt(usb1PidLabel, "PID: 0x%lx", pid);
        } else {
            lv_label_set_text_fmt(usb1VidLabel, "VID: ------");
            lv_label_set_text_fmt(usb1PidLabel, "PID: ------");
        }
    } else if(usbId == 1) {
        if(attached) {
            lv_label_set_text_fmt(usb2VidLabel, "VID: 0x%lx", vid);
            lv_label_set_text_fmt(usb2PidLabel, "PID: 0x%lx", pid);
        } else {
            lv_label_set_text_fmt(usb2VidLabel, "VID: ------");
            lv_label_set_text_fmt(usb2PidLabel, "PID: ------");
        }
    }
}

void guiUpdateRs485Ping(uint8_t state) {
    if(state) {
        pingRs485Counter++;
        lv_label_set_text_fmt(pingRs485Label, "PING: %lu", pingRs485Counter);
    } else {
        pingRs485Counter = 0;
        lv_label_set_text(pingRs485Label, "PING: FAILED");
    }
}

void guiUpdateCanPing(uint8_t state) {
    if(state) {
        pingCanCounter++;
        lv_label_set_text_fmt(pingCanLabel, "PING: %lu", pingCanCounter);
    } else {
        pingCanCounter = 0;
        lv_label_set_text(pingCanLabel, "PING: FAILED");
    }
}

void guiUpdateEthPing(uint8_t state, uint8_t ms) {
    if(state) {
        pingEthCounter++;
        lv_label_set_text_fmt(pingEthLabel, "PING: %lu [%d ms]", pingEthCounter, ms);
    } else {
        pingEthCounter = 0;
        lv_label_set_text(pingEthLabel, "PING: FAILED");
    }
}
