/*
 * gpioTest.c
 *
 *  Created on: Mar 7, 2025
 *      Author: krzysiek
 */

#include <stdio.h>
#include "main.h"
#include "gpioTest.h"

extern ADC_HandleTypeDef hadc1;
extern ADC_HandleTypeDef hadc3;

typedef struct {
    GPIO_TypeDef* baseIn;
    uint32_t pinIn;
    bool isInAnalog;
    ADC_HandleTypeDef* hadc;
    uint32_t adcChannel;
    GPIO_TypeDef* baseOut;
    uint32_t pinOut;
    char* name;
} GpioTestPair_t;

GpioTestPair_t gpioTestPairs[] = {
#if IS_WIFI_MODULE == 0
        {.baseIn = GPIOD, .pinIn = GPIO_PIN_6, .isInAnalog = false,
         .baseOut = GPIOD, .pinOut = GPIO_PIN_5,
         .name = "UART2.RX (PD6) - UART2.TX (PD5)"},

        {.baseIn = GPIOD, .pinIn = GPIO_PIN_3, .isInAnalog = false,
         .baseOut = GPIOD, .pinOut = GPIO_PIN_4,
         .name = "UART2.CTS (PD3) - UART2.RTS (PD4)"},

        {.baseIn = GPIOI, .pinIn = GPIO_PIN_8, .isInAnalog = false,
         .baseOut = GPIOC, .pinOut = GPIO_PIN_7,
         .name = "PORTI.8 (PI8) - PORTC.7 (PC7)"},

        {.baseIn = GPIOI, .pinIn = GPIO_PIN_10, .isInAnalog = false,
         .baseOut = GPIOI, .pinOut = GPIO_PIN_9,
         .name = "PORTI.10 (PI10) - PORTI.9 (PI9)"},
#endif
        {.baseIn = GPIOB, .pinIn = GPIO_PIN_5, .isInAnalog = false,
         .baseOut = GPIOF, .pinOut = GPIO_PIN_9,
         .name = "PORTB.5 (PB5) - UART7.CTS (PF9)"},

        {.baseIn = GPIOC, .pinIn = GPIO_PIN_2, .isInAnalog = true,
         .hadc = &hadc3, .adcChannel = ADC_CHANNEL_0,
         .baseOut = GPIOF, .pinOut = GPIO_PIN_9,
         .name = "PC2_C (PC2_C) - UART7.CTS (PF9)"},

        {.baseIn = GPIOC, .pinIn = GPIO_PIN_3, .isInAnalog = true,
         .hadc = &hadc3, .adcChannel = ADC_CHANNEL_1,
         .baseOut = GPIOF, .pinOut = GPIO_PIN_8,
         .name = "PC3_C (PC3_C) - UART7.RTS (PF8)"},

        {.baseIn = GPIOA, .pinIn = GPIO_PIN_0, .isInAnalog = true,
         .hadc = &hadc1, .adcChannel = ADC_CHANNEL_0,
         .baseOut = GPIOF, .pinOut = GPIO_PIN_6,
         .name = "PA0_C (PA0_C) - UART7.RXD (PF6)"},

        {.baseIn = GPIOA, .pinIn = GPIO_PIN_1, .isInAnalog = true,
         .hadc = &hadc1, .adcChannel = ADC_CHANNEL_1,
         .baseOut = GPIOF, .pinOut = GPIO_PIN_7,
         .name = "PA1_C (PA1_C) - UART7.TXD (PF7)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_12, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_13,
         .name = "LCD.B0 (PJ12) - LCD.B1 (PJ13)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_14, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_15,
         .name = "LCD.B2 (PJ14) - LCD.B3 (PJ15)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_7, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_8,
         .name = "LCD.G0 (PJ7) - LCD.G1 (PJ8)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_9, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_10,
         .name = "LCD.G2 (PJ9) - LCD.G3 (PJ10)"},

        {.baseIn = GPIOI, .pinIn = GPIO_PIN_15, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_0,
         .name = "LCD.R0 (PI15) - LCD.R1 (PJ0)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_1, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_2,
         .name = "LCD.R2 (PJ1) - LCD.R3 (PJ2)"},
#if IS_MEM_MODULE == 0
        {.baseIn = GPIOK, .pinIn = GPIO_PIN_3, .isInAnalog = false,
         .baseOut = GPIOK, .pinOut = GPIO_PIN_4,
         .name = "LCD.B4 (PK3) - LCD.B5 (PK4)"},

        {.baseIn = GPIOK, .pinIn = GPIO_PIN_5, .isInAnalog = false,
         .baseOut = GPIOK, .pinOut = GPIO_PIN_6,
         .name = "LCD.B6 (PK5) - LCD.B7 (PK6)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_11, .isInAnalog = false,
         .baseOut = GPIOK, .pinOut = GPIO_PIN_0,
         .name = "LCD.G4 (PJ11) - LCD.G5 (PK0)"},

        {.baseIn = GPIOK, .pinIn = GPIO_PIN_1, .isInAnalog = false,
         .baseOut = GPIOK, .pinOut = GPIO_PIN_2,
         .name = "LCD.G6 (PK1) - LCD.G7 (PK2)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_3, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_4,
         .name = "LCD.R4 (PJ3) - LCD.R5 (PJ4)"},

        {.baseIn = GPIOJ, .pinIn = GPIO_PIN_5, .isInAnalog = false,
         .baseOut = GPIOJ, .pinOut = GPIO_PIN_6,
         .name = "LCD.R6 (PJ5) - LCD.R7 (PJ6)"},

        {.baseIn = GPIOK, .pinIn = GPIO_PIN_7, .isInAnalog = false,
         .baseOut = GPIOI, .pinOut = GPIO_PIN_12,
         .name = "LCD.DE (PK7) - LCD.HSYNC (PI12)"},

        {.baseIn = GPIOI, .pinIn = GPIO_PIN_13, .isInAnalog = false,
         .baseOut = GPIOG, .pinOut = GPIO_PIN_7,
         .name = "LCD.VSYNC (PI13) - LCD.CLK (PG7)"},

        {.baseIn = GPIOB, .pinIn = GPIO_PIN_10, .isInAnalog = false,
         .baseOut = GPIOH, .pinOut = GPIO_PIN_11,
         .name = "PORTB.10 (PB10) - PORTH.11 (PH11)"},

        {.baseIn = GPIOB, .pinIn = GPIO_PIN_11, .isInAnalog = false,
         .baseOut = GPIOH, .pinOut = GPIO_PIN_6,
         .name = "PORTB.11 (PB11) - PORTH.6 (PH6)"},

        {.baseIn = GPIOH, .pinIn = GPIO_PIN_7, .isInAnalog = false,
         .baseOut = GPIOB, .pinOut = GPIO_PIN_0,
         .name = "I2C3.SCL (PH7) - PORTB.0 (PB0)"},

        {.baseIn = GPIOH, .pinIn = GPIO_PIN_8, .isInAnalog = false,
         .baseOut = GPIOB, .pinOut = GPIO_PIN_0,
         .name = "I2C3.SDA (PH8) - PORTB.0 (PB0)"},

        {.baseIn = GPIOI, .pinIn = GPIO_PIN_5, .isInAnalog = false,
         .baseOut = GPIOA, .pinOut = GPIO_PIN_4,
         .name = "DCAM.VSYNC (PI5) - DCAM.HSYNC (PA4)"},

        {.baseIn = GPIOA, .pinIn = GPIO_PIN_6, .isInAnalog = false,
         .baseOut = GPIOI, .pinOut = GPIO_PIN_6,
         .name = "DCAM.PCLK (PA6) - DCAM.D6 (PI6)"},

        {.baseIn = GPIOI, .pinIn = GPIO_PIN_7, .isInAnalog = false,
         .baseOut = GPIOE, .pinOut = GPIO_PIN_4,
         .name = "DCAM.D7 (PI7) - DCAM.D4 (PE4)"},

        {.baseIn = GPIOI, .pinIn = GPIO_PIN_4, .isInAnalog = false,
         .baseOut = GPIOG, .pinOut = GPIO_PIN_10,
         .name = "DCAM.D5 (PI4) - DCAM.D2 (PG10)"},

        {.baseIn = GPIOH, .pinIn = GPIO_PIN_12, .isInAnalog = false,
         .baseOut = GPIOH, .pinOut = GPIO_PIN_9,
         .name = "DCAM.D3 (PH12) - DCAM.D0 (PH9)"},

        {.baseIn = GPIOH, .pinIn = GPIO_PIN_10, .isInAnalog = false,
         .baseOut = GPIOE, .pinOut = GPIO_PIN_3,
         .name = "DCAM.D1 (PH10) - CAM-RST (PE3)"},

#endif
};

static int readAnalogInput(GpioTestPair_t* testPair) {
    ADC_ChannelConfTypeDef sConfig = {0};
    sConfig.Channel = testPair->adcChannel;
    sConfig.Rank = ADC_REGULAR_RANK_1;
    sConfig.SamplingTime = ADC_SAMPLETIME_1CYCLE_5;
    sConfig.SingleDiff = ADC_SINGLE_ENDED;
    sConfig.OffsetNumber = ADC_OFFSET_NONE;
    sConfig.Offset = 0;
    sConfig.OffsetSignedSaturation = DISABLE;
    HAL_ADC_ConfigChannel(testPair->hadc, &sConfig);

    HAL_ADC_Start(testPair->hadc);
    HAL_ADC_PollForConversion(testPair->hadc, 100);
    uint32_t value = HAL_ADC_GetValue(testPair->hadc);
    HAL_ADC_Stop(testPair->hadc);

    if(value > 65000)
      return 1;
    else if(value < 500)
      return 0;
    else
      return -1;
}

bool gpioTest(void) {
    GPIO_InitTypeDef GPIO_InitStructInput = {0};
    GPIO_InitStructInput.Mode = GPIO_MODE_INPUT;
    GPIO_InitStructInput.Pull = GPIO_NOPULL;

    GPIO_InitTypeDef GPIO_InitStructOutput = {0};
    GPIO_InitStructOutput.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_InitStructOutput.Pull = GPIO_PULLDOWN;
    GPIO_InitStructOutput.Speed = GPIO_SPEED_FREQ_LOW;

    bool failed = false;

    for(int i = 0; i < sizeof(gpioTestPairs)/sizeof(gpioTestPairs[0]); i++) {
        if(!gpioTestPairs[i].isInAnalog) {
            GPIO_InitStructInput.Pin = gpioTestPairs[i].pinIn;
            HAL_GPIO_Init(gpioTestPairs[i].baseIn, &GPIO_InitStructInput);
        }

        GPIO_InitStructOutput.Pin = gpioTestPairs[i].pinOut;
        HAL_GPIO_Init(gpioTestPairs[i].baseOut, &GPIO_InitStructOutput);
    }


    for(int i = 0; i < sizeof(gpioTestPairs)/sizeof(gpioTestPairs[0]); i++) {

        HAL_GPIO_WritePin(gpioTestPairs[i].baseOut, gpioTestPairs[i].pinOut, GPIO_PIN_RESET);

        if(gpioTestPairs[i].isInAnalog) {
          int input = readAnalogInput(&gpioTestPairs[i]);
          if(input != 0) {
            printf("GPIO test failed (0): %s\r\n", gpioTestPairs[i].name);
            failed = true;
          }
        } else {
            if(HAL_GPIO_ReadPin(gpioTestPairs[i].baseIn, gpioTestPairs[i].pinIn) != GPIO_PIN_RESET) {
                printf("GPIO test failed (0): %s\r\n", gpioTestPairs[i].name);
                failed = true;
            }
        }

        HAL_GPIO_WritePin(gpioTestPairs[i].baseOut, gpioTestPairs[i].pinOut, GPIO_PIN_SET);

        if(gpioTestPairs[i].isInAnalog) {
          int input = readAnalogInput(&gpioTestPairs[i]);
          if(input != 1) {
              printf("GPIO test failed (1): %s\r\n", gpioTestPairs[i].name);
              failed = true;
          }
        } else {
            if(HAL_GPIO_ReadPin(gpioTestPairs[i].baseIn, gpioTestPairs[i].pinIn) != GPIO_PIN_SET) {
                printf("GPIO test failed (1): %s\r\n", gpioTestPairs[i].name);
                failed = true;
            }
        }

        HAL_GPIO_WritePin(gpioTestPairs[i].baseOut, gpioTestPairs[i].pinOut, GPIO_PIN_RESET);
    }

    return !failed;
}
