/*
 * cameraTask.c
 *
 *  Created on: Feb 13, 2025
 *      Author: krzysiek
 */

#include "cameraTask.h"
#include "ov5640.h"
#include "test.h"
#include "guiTask.h"
#include "gui.h"

#define OV5640_I2C_TIMEOUT_DEFAULT  1000
#define OV5640_I2C_ADDRESS  0x3C
#define CAMERA_OV5640_ADDRESS (OV5640_I2C_ADDRESS << 1)

static OV5640_Capabilities_t Camera_Cap;
static OV5640_Object_t OV5640Obj;
static uint32_t CameraId;

osThreadId_t cameraTaskHandle;

const osThreadAttr_t cameraTask_attributes = {
  .name = "cameraTask",
  .stack_size = 512 * 4,
  .priority = (osPriority_t) osPriorityLow,
};

extern I2C_HandleTypeDef hi2c3;
extern DCMI_HandleTypeDef hdcmi;


static int32_t BSP_I2C3_ReadReg16(uint16_t DevAddr, uint16_t Reg, uint8_t *pData, uint16_t Length)
{
  HAL_StatusTypeDef result;

  result = HAL_I2C_Mem_Read(&hi2c3, DevAddr, Reg, I2C_MEMADD_SIZE_16BIT, pData, Length, OV5640_I2C_TIMEOUT_DEFAULT);

  return (result == HAL_OK)?0:1;
}

static int32_t BSP_I2C3_WriteReg16(uint16_t DevAddr, uint16_t Reg, uint8_t *pData, uint16_t Length)
{
  HAL_StatusTypeDef result;

  result = HAL_I2C_Mem_Write(&hi2c3, DevAddr, Reg, I2C_MEMADD_SIZE_16BIT, pData, Length, OV5640_I2C_TIMEOUT_DEFAULT);

  return (result == HAL_OK)?0:1;
}

static int32_t BSP_GetTick(void)
{
  return (int32_t)HAL_GetTick();
}

static int32_t BSP_I2C3_Init(void)
{
  return 0;
}

static int32_t BSP_I2C3_DeInit(void)
{
  return 0;
}

static uint32_t OV5640_Probe(uint32_t resolution, uint32_t pixelFormat)
{
  int32_t ret;
  OV5640_IO_t              IOCtx;

  /* Configure the audio driver */
  IOCtx.Address     = CAMERA_OV5640_ADDRESS;
  IOCtx.Init        = BSP_I2C3_Init;
  IOCtx.DeInit      = BSP_I2C3_DeInit;
  IOCtx.ReadReg     = BSP_I2C3_ReadReg16;
  IOCtx.WriteReg    = BSP_I2C3_WriteReg16;
  IOCtx.GetTick     = BSP_GetTick;

  if(OV5640_RegisterBusIO (&OV5640Obj, &IOCtx) != OV5640_OK) {
    ret = 1;
  }else if(OV5640_ReadID(&OV5640Obj, &CameraId) != OV5640_OK) {
   ret = 1;
  } else {
    if(CameraId != OV5640_ID) {
      ret = 2;
    } else {
      if(OV5640_CAMERA_Driver.Init(&OV5640Obj, resolution, pixelFormat) != OV5640_OK) {
        ret = 1;
      } else if(OV5640_CAMERA_Driver.GetCapabilities(&OV5640Obj, &Camera_Cap) != OV5640_OK) {
        ret = 1;
      } else {
        ret = 0;
      }
    }
  }

  //OV5640_CAMERA_Driver.SetBrightness(&OV5640Obj, 4);

  return ret;
}

static uint32_t GetCameraBufferSize(uint32_t Resolution, uint32_t PixelFormat)
{
  uint32_t size = 0;
  uint32_t pf_div;
  pf_div = (PixelFormat == OV5640_RGB888) ? 3 : 2;

  /* Get capture size */
  switch (Resolution) {
    case OV5640_R160x120:
      size = ((uint32_t)(160*120)*pf_div)/4U;
      break;
    case OV5640_R320x240:
      size = ((uint32_t)(320*240)*pf_div)/4U;
      break;
    case OV5640_R480x272:
      size =  ((uint32_t)(480*272)*pf_div)/4U;
      break;
    case OV5640_R640x480:
      size =  ((uint32_t)(640*480)*pf_div)/4U;
      break;
    case OV5640_R800x480:
      size =  ((uint32_t)(800*480)*pf_div)/4U;
      break;
    default:
      break;
  }

  return size;
}

void cameraTask(void *argument) {

    OV5640_Probe(OV5640_R480x272, OV5640_RGB565);

    while(1) {
        HAL_DCMI_Start_DMA(&hdcmi, DCMI_MODE_SNAPSHOT, (uint32_t)imageBuffer, GetCameraBufferSize(OV5640_R480x272, OV5640_RGB565));
        vTaskDelay(400 / portTICK_PERIOD_MS);
        if(hdcmi.State != HAL_DCMI_STATE_READY) {
            HAL_DCMI_Stop(&hdcmi);
            continue;
        }

        GuiEvent_t event;
        event.type = GuiEventType_Camera;
        osMessageQueuePut(guiEventQueueHandle, &event, 0, 0);
    }
}
