#include "cmsis_os.h"
#include "ledTask.h"

#include "lvgl/lvgl.h"
#include "lvgl_port_display.h"
#include "test.h"
#include "gui.h"

#define LEDS_COUNT 4
#define FANS_COUNT 10

typedef struct {
    uint8_t ledId;
    uint8_t stateId;
} LedData_t;

typedef struct {
    uint8_t active;
    lv_obj_t* activeImage;
    lv_obj_t* inactiveImage;
} FanState_t;

static FanState_t fans[FANS_COUNT];

static LedData_t led1Data = {.ledId = 0};
static LedData_t led2Data = {.ledId = 1};
static LedData_t led3Data = {.ledId = 2};
static LedData_t led4Data = {.ledId = 3};

static lv_obj_t* leds[LEDS_COUNT];
static lv_obj_t* sliderLabel;
static lv_obj_t* scale;
static lv_obj_t* needleLine;

extern lv_img_dsc_t background_pic;
extern lv_img_dsc_t icon_light_pic;
extern lv_img_dsc_t icon_light_off_pic;
extern lv_img_dsc_t icon_fan_active_pic;
extern lv_img_dsc_t icon_fan_inactive_pic;
extern lv_img_dsc_t som_pic;

__attribute__((section("DisplayMemNonCacheable"), aligned(64)))
uint8_t imageBuffer[IMAGE_BUFFER_HEIGHT * IMAGE_BUFFER_WIDTH * IMAGE_BUFFER_BPP];

static const lv_img_dsc_t imageData = {
    .header.w = IMAGE_BUFFER_WIDTH,
    .header.h = IMAGE_BUFFER_HEIGHT,
    .header.cf = LV_COLOR_FORMAT_NATIVE,
    .data = imageBuffer,
    .data_size = IMAGE_BUFFER_HEIGHT * IMAGE_BUFFER_WIDTH * IMAGE_BUFFER_BPP,
};

static lv_obj_t* cameraImage;

static void ledButtonEventHandler(lv_event_t* e)
{
    LedData_t* ledData = lv_event_get_user_data(e);
    lv_obj_t* button = lv_event_get_target(e);

    LedEvent_t event;
    event.ledId = ledData->ledId;

    if (lv_obj_has_state(button, LV_STATE_CHECKED) == true) {
        event.ledState = LedState_On;

    } else {
        event.ledState = LedState_Off;
    }

    osMessageQueuePut(ledQueueHandle, &event, 0, 0);
}

static void initCameraTab(lv_obj_t* parent) {
    cameraImage = lv_img_create(parent);
    lv_img_set_src(cameraImage, &imageData);
    lv_obj_center(cameraImage);
}

static void ledBoxCreate(lv_obj_t* parent, uint32_t y) {
    for(int i = 0; i < LEDS_COUNT; i++) {
        leds[i] = lv_led_create(parent);
        lv_led_set_color(leds[i], lv_palette_main(LV_PALETTE_RED));
        lv_led_off(leds[i]);
        lv_obj_set_pos(leds[i], 85 + (i * 256), y);
        lv_obj_set_size(leds[i], 50, 50);
    }
}

static void ledButtonCreate(lv_obj_t* parent, uint32_t x, uint32_t y, LedData_t* ledData) {
    lv_obj_t* ledButton = lv_imagebutton_create(parent);
    lv_imagebutton_set_src(ledButton, LV_IMAGEBUTTON_STATE_CHECKED_RELEASED, NULL, &icon_light_pic, NULL);
    lv_imagebutton_set_src(ledButton, LV_IMAGEBUTTON_STATE_PRESSED, NULL, &icon_light_pic, NULL);
    lv_imagebutton_set_src(ledButton, LV_IMAGEBUTTON_STATE_CHECKED_PRESSED, NULL, &icon_light_off_pic, NULL);
    lv_imagebutton_set_src(ledButton, LV_IMAGEBUTTON_STATE_RELEASED, NULL, &icon_light_off_pic, NULL);

    lv_obj_add_flag(ledButton, LV_OBJ_FLAG_CHECKABLE);
    lv_obj_set_pos(ledButton, x, y);
    lv_obj_set_size(ledButton, icon_light_pic.header.w, icon_light_pic.header.h);
    lv_obj_add_event_cb(ledButton, ledButtonEventHandler, LV_EVENT_VALUE_CHANGED, ledData);
}

static void initLedTab(lv_obj_t* parent) {
    ledButtonCreate(parent, 60, 150, &led1Data);
    ledButtonCreate(parent, 60 + 256, 150, &led2Data);
    ledButtonCreate(parent, 60 + 512, 150, &led3Data);
    ledButtonCreate(parent, 60 + 768, 150, &led4Data);

    ledBoxCreate(parent, 350);
}

static void sliderEventHandler(lv_event_t* e)
{
    lv_obj_t* slider = lv_event_get_target(e);
    char buf[16];
    int value = lv_slider_get_value(slider);
    lv_snprintf(buf, sizeof(buf), "Fan speed %d%%", value);
    lv_label_set_text(sliderLabel, buf);
    lv_obj_align_to(sliderLabel, slider, LV_ALIGN_OUT_BOTTOM_MID, 0, 10);

    lv_scale_set_line_needle_value(scale, needleLine, 100, value);

    int activeFans = value / 10 + 1;
    if(value == 0)
        activeFans = 0;
    if(value == 100)
        activeFans = 10;

    for(int i = 0; i < activeFans; i++) {
        if(fans[i].active == 0) {
            fans[i].active = 1;
            lv_obj_clear_flag(fans[i].activeImage, LV_OBJ_FLAG_HIDDEN);
            lv_obj_add_flag(fans[i].inactiveImage, LV_OBJ_FLAG_HIDDEN);
        }
    }

    for(int i = activeFans; i < FANS_COUNT; i++) {
        if(fans[i].active == 1) {
            fans[i].active = 0;
            lv_obj_add_flag(fans[i].activeImage, LV_OBJ_FLAG_HIDDEN);
            lv_obj_clear_flag(fans[i].inactiveImage, LV_OBJ_FLAG_HIDDEN);
        }
    }
}

static void initFanTab(lv_obj_t* parent) {

    lv_obj_t * slider = lv_slider_create(parent);
    lv_obj_align(slider, LV_ALIGN_TOP_MID, 0, 450);
    lv_obj_set_size(slider, 400, 20);
    lv_obj_add_event_cb(slider, sliderEventHandler, LV_EVENT_VALUE_CHANGED, NULL);

    sliderLabel = lv_label_create(parent);
    lv_label_set_text(sliderLabel, "Fan speed 0%");
    lv_obj_align_to(sliderLabel, slider, LV_ALIGN_OUT_BOTTOM_MID, 0, 10);
    lv_obj_set_style_text_color(sliderLabel, lv_color_hex(0xFFFFFF), 0);

    scale = lv_scale_create(parent);

    lv_obj_set_size(scale, 300, 300);
    lv_scale_set_mode(scale, LV_SCALE_MODE_ROUND_INNER);
    lv_obj_set_style_bg_opa(scale, LV_OPA_TRANSP, 0);
    //lv_obj_set_style_bg_color(scale, lv_palette_lighten(LV_PALETTE_RED, 5), 0);
    //lv_obj_set_style_radius(scale, LV_RADIUS_CIRCLE, 0);
    //lv_obj_set_style_clip_corner(scale, true, 0);
    lv_obj_align(scale, LV_ALIGN_TOP_MID, 0, 75);

    lv_scale_set_label_show(scale, true);

    lv_scale_set_total_tick_count(scale, 21);
    lv_scale_set_major_tick_every(scale, 5);

    lv_obj_set_style_length(scale, 5, LV_PART_ITEMS);
    lv_obj_set_style_length(scale, 10, LV_PART_INDICATOR);
    lv_scale_set_range(scale, 0, 100);

    needleLine = lv_line_create(scale);

    lv_obj_set_style_line_width(needleLine, 4, LV_PART_MAIN);
    lv_obj_set_style_line_rounded(needleLine, true, LV_PART_MAIN);

    lv_obj_set_style_line_color(needleLine, lv_color_hex(0xFFFFFF), LV_PART_MAIN);

    lv_obj_set_style_text_color(scale, lv_color_hex(0xFFFFFF), LV_PART_INDICATOR);
    lv_obj_set_style_arc_color(scale, lv_color_hex(0xFFFFFF), LV_PART_MAIN);
    lv_obj_set_style_line_color(scale, lv_color_hex(0xFFFFFF), LV_PART_INDICATOR);
    lv_obj_set_style_line_color(scale, lv_color_hex(0xFFFFFF), LV_PART_ITEMS);

    lv_scale_set_line_needle_value(scale, needleLine, 100, 0);

    for(int i = 0; i < FANS_COUNT; i++) {
        fans[i].active = 0;
        fans[i].activeImage = lv_img_create(parent);
        fans[i].inactiveImage = lv_img_create(parent);
        lv_img_set_src(fans[i].activeImage, &icon_fan_active_pic);
        lv_img_set_src(fans[i].inactiveImage, &icon_fan_inactive_pic);
        lv_obj_align(fans[i].activeImage, LV_ALIGN_TOP_MID, (i - FANS_COUNT / 2) * 40 + 20, 380);
        lv_obj_align(fans[i].inactiveImage, LV_ALIGN_TOP_MID, (i - FANS_COUNT / 2) * 40 + 20, 380);
        lv_obj_add_flag(fans[i].activeImage, LV_OBJ_FLAG_HIDDEN);
    }

}

static void addTextLine(lv_obj_t* parent, const char* text, int x, int y) {
    lv_obj_t* label = lv_label_create(parent);
    lv_label_set_text(label, text);
    lv_obj_set_pos(label, x, y);
    lv_obj_set_style_text_color(label, lv_color_hex(0xFFFFFF), 0);
}

static void initAboutTab(lv_obj_t* parent) {
    lv_obj_t* somImg = lv_img_create(parent);
    lv_img_set_src(somImg, &som_pic);
    lv_obj_align(somImg, LV_ALIGN_CENTER, -300, 0);

    addTextLine(parent, "StarSOM-STM32H757 + VisionCB-STM32H757-STD", 350, 120);
    addTextLine(parent, "STM32H757 dual core Cortex-M7+Cortex-M4", 350, 160);
    addTextLine(parent, "1MB on-chip RAM, up to 32MB SDRAM", 350, 200);
    addTextLine(parent, "2MB on-chip Flash, up to 16MB QSPI Flash", 350, 240);
    addTextLine(parent, "OV5640 camera module", 350, 280);
    addTextLine(parent, "SL-TFT7-TP-600-1024-MIPI display", 350, 320);
    addTextLine(parent, "Demo based on ST CubeMX SDK and LVGL library", 350, 360);
}

void guiInit(void) {

    lv_obj_set_style_bg_img_src(lv_scr_act(), &background_pic, 0);

    lv_obj_t* tabview = lv_tabview_create(lv_scr_act());
    lv_tabview_set_tab_bar_position(tabview, LV_DIR_BOTTOM);
    lv_tabview_set_tab_bar_size(tabview, 50);
    lv_obj_set_style_bg_opa(tabview, LV_OPA_TRANSP, 0);

    lv_obj_t* ledTab = lv_tabview_add_tab(tabview, "Light");
    lv_obj_t* fanTab = lv_tabview_add_tab(tabview, "Fan");
    lv_obj_t* cameraTab = lv_tabview_add_tab(tabview, "Camera");
    lv_obj_t* aboutTab = lv_tabview_add_tab(tabview, "About");

    initCameraTab(cameraTab);
    initLedTab(ledTab);
    initFanTab(fanTab);
    initAboutTab(aboutTab);
}

void guiSetLedState(uint8_t id, uint8_t state) {
    if(id > LEDS_COUNT)
        return;

    if(state == 0)
        lv_led_off(leds[id]);
    else
        lv_led_set_brightness(leds[id], 255);
}

void guiUpdateCameraImage(void) {
    lv_obj_invalidate(cameraImage);
}
